package cn.com.duibaboot.ext.autoconfigure.data.redis;

import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;

import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.wolf.cache.Hessian2SerializationRedisSerializer;
import io.lettuce.core.RedisClient;

import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionBuilder;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.ImportBeanDefinitionRegistrar;
import org.springframework.core.env.Environment;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.StringRedisTemplate;

/**
 * 注入额外的RedisTemplate
 */
//也可以用BeanDefinitionRegistryPostProcessor来注入BeanDefinition
@ConditionalOnClass(RedisClient.class)
public class ExtraRedisTemplateImportBeanDefinitionRegistrar4Lettuce implements ImportBeanDefinitionRegistrar, EnvironmentAware {

    private List<String> extraRedisTemplateIdsPrefix = new ArrayList<>();

    @Override
    public void registerBeanDefinitions(AnnotationMetadata importingClassMetadata, BeanDefinitionRegistry registry) {
        for(String extraRedisTemplateIdPrefix : extraRedisTemplateIdsPrefix) {
            String redisConnectionFactoryBeanId = extraRedisTemplateIdPrefix + "RedisConnectionFactory";
            if(registry.containsBeanDefinition(redisConnectionFactoryBeanId)){
                //已经被ExtraRedisTemplateImportBeanDefinitionRegistrar4Jedis注册过，跳过
                break;
            }
            BeanDefinition extraRedisConnectionFactoryDefinition = BeanDefinitionBuilder
                    .genericBeanDefinition(LettuceConnectionFactoryBean.class)
                    .addConstructorArgValue(extraRedisTemplateIdPrefix)
                    .getBeanDefinition();
            registry.registerBeanDefinition(redisConnectionFactoryBeanId,
                    extraRedisConnectionFactoryDefinition);

            BeanDefinition extraRedisTemplateDefinition = BeanDefinitionBuilder
                    .genericBeanDefinition(RedisTemplate.class)
                    .addPropertyReference("connectionFactory", redisConnectionFactoryBeanId)
                    //使用hession2序列化方式，提高性能，减少占用redis的空间
                    .addPropertyValue("defaultSerializer", new Hessian2SerializationRedisSerializer())
                    .getBeanDefinition();
            registry.registerBeanDefinition(extraRedisTemplateIdPrefix + "RedisTemplate",
                    extraRedisTemplateDefinition);

            BeanDefinition extraStringRedisTemplateDefinition = BeanDefinitionBuilder
                    .genericBeanDefinition(StringRedisTemplate.class)
                    .addPropertyReference("connectionFactory", redisConnectionFactoryBeanId)
                    .getBeanDefinition();
            registry.registerBeanDefinition(extraRedisTemplateIdPrefix + "StringRedisTemplate",
                    extraStringRedisTemplateDefinition);
        }
    }

    @Override
    public void setEnvironment(Environment environment) {
        String redisExtraIdPrefix = "duiba.redis.extra.";
        String hbaseExtraIdPostfix = ".host";
        LinkedHashMap<String, Object> props = SpringEnvironmentUtils.getFlatEnvironments(environment);
        props.entrySet().stream()
                .filter(e -> e.getKey().startsWith(redisExtraIdPrefix) && e.getKey().endsWith(hbaseExtraIdPostfix))
                .map(e -> e.getKey().substring(redisExtraIdPrefix.length(), e.getKey().length() - hbaseExtraIdPostfix.length()))
                .forEach(s -> extraRedisTemplateIdsPrefix.add(s));
    }

}
