package cn.com.duibaboot.ext.autoconfigure.flowreplay;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;
import org.springframework.http.ResponseEntity;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.io.*;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import java.util.zip.ZipOutputStream;

/**
 * 引流回归平台使用，文件上传和下载的组件
 * Created by guoyanfei .
 * 2019-02-22 .
 */
@Slf4j
public class FlowReplayFileComponent {

    private final RestTemplate loadBalancedRestTemplate;

    public FlowReplayFileComponent(RestTemplate loadBalancedRestTemplate) {
        this.loadBalancedRestTemplate = loadBalancedRestTemplate;
    }

    /**
     * 文件上传到oss
     * 1、根据本地文件路径把文件压缩
     * 2、调用athena-web的上传接口上传，返回oss地址
     *
     * @param localPath 可以是一个文件，也可以是一个目录。如果是目录，按照目录原有的结构进行构造
     * @return
     */
    public String upload(String localPath) throws IOException {
        String localZipFileName = FlowReplayConstants.LOCAL_FLOWREPLAY_DIR + System.currentTimeMillis() + ".zip";
        File localFile = new File(localPath);
        File localZipFile = new File(localZipFileName);

        // 压缩文件localFile，创建新的文件localZipFile
        try (OutputStream os = new FileOutputStream(localZipFile); ZipOutputStream zipos = new ZipOutputStream(os)) {
            compressFileToZip(localFile, zipos, localFile.getName());
        }

        // 压缩好的文件localZipFile，通过athena-web上传到oss中
        FileSystemResource resource = new FileSystemResource(localZipFile);
        MultiValueMap<String, Object> multiValueMap = new LinkedMultiValueMap<>();
        multiValueMap.add("file", resource);

        return loadBalancedRestTemplate.postForObject("http://athena-web/athena/file/upload", multiValueMap, String.class);
    }

    /**
     * 压缩文件成一个zip，支持压缩目录
     * @param sourceFile
     * @param zipos
     * @param fileName
     * @throws IOException
     */
    private void compressFileToZip(File sourceFile, ZipOutputStream zipos, String fileName) throws IOException {
        byte[] buffer = new byte[1024];
        if (sourceFile.isFile()) {        // 来源文件是个具体文件，直接加入压缩包中
            zipos.putNextEntry(new ZipEntry(fileName));
            try (InputStream is = new FileInputStream(sourceFile); BufferedInputStream bis = new BufferedInputStream(is)) {
                int i = bis.read(buffer);
                while (i != -1) {
                    zipos.write(buffer, 0, i);
                    i = bis.read(buffer);
                }
            }
            zipos.closeEntry();
        } else {    // 来源文件是个目录，递归
            File[] listFiles = sourceFile.listFiles();
            if (listFiles == null || listFiles.length == 0) {
                zipos.putNextEntry(new ZipEntry(fileName + "/"));
                zipos.closeEntry();
            } else {
                for (File subFile : listFiles) {
                    compressFileToZip(subFile, zipos, fileName + "/" + subFile.getName());
                }
            }
        }

    }

    /**
     * 把需要回归的文件下载下来
     * 1、根据oss地址，调用athena-web的下载接口下载
     * 2、把下载的文件解压缩
     * @param ossPath oss的文件路径
     * @param localDir 文件下载后本地存储目录
     * @return
     * @throws IOException
     */
    public String download(String ossPath, String localDir) throws IOException {
        if (StringUtils.isBlank(ossPath) || StringUtils.isBlank(localDir)) {
            throw new FlowReplayException("文件下载参数不全");
        }
        ResponseEntity<Resource> response = loadBalancedRestTemplate.getForEntity("http://athena-web/athena/file/download?objectName=" + ossPath, Resource.class);
        InputStream input = response.getBody().getInputStream();

        // 把下载的zip文件localZipFile解压缩到newFilePath
        String newFilePath = ((localDir.endsWith("/") ? localDir : localDir + "/") + System.currentTimeMillis());
        try (ZipInputStream zipInputStream = new ZipInputStream(input);
                FileOutputStream fileOutputStream = new FileOutputStream(newFilePath)) {
            zipInputStream.getNextEntry();
            byte[] buf = new byte[1024];
            int len;
            while ((len = zipInputStream.read(buf, 0, 1024)) != -1) {
                fileOutputStream.write(buf, 0, len);
            }
        }
        return newFilePath;
    }

    /**
     * 清空路径文件的内容
     * @param filePath
     */
    public void clearFile(String filePath) {
        File file = new File(filePath);
        if (!file.exists() || file.isDirectory()) {
            return;
        }
        try (FileWriter fw = new FileWriter(file)) {
            fw.write("");
            fw.flush();
        } catch (IOException e) {
            log.error("文件清空异常,filePath={}", filePath, e);
        }
    }

    /**
     * 初始化引流回归的本地文件目录 /data/flowreplay/
     * 删除，并重新创建目录
     */
    public static void initDir() {
        File dir = new File(FlowReplayConstants.LOCAL_FLOWREPLAY_DIR);
        try {
            FileUtils.deleteDirectory(dir);
            FileUtils.forceMkdir(dir);
        } catch (IOException e) {
            log.error("初始化/data/flowreplay目录异常", e);
        }
    }

    /**
     * 删除引流回归的本地文件目录 /data/flowreplay/
     */
    public static void removeDir() {
        File dir = new File(FlowReplayConstants.LOCAL_FLOWREPLAY_DIR);
        try {
            FileUtils.deleteDirectory(dir);
        } catch (IOException e) {
            log.error("初始化/data/flowreplay目录异常", e);
        }
    }
}
