package cn.com.duibaboot.ext.autoconfigure.flowreplay.record;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayException;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.FlowReplayTrace;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.RecordDetailContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.endpoint.RecordConfigDto;
import lombok.extern.slf4j.Slf4j;

import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;

/**
 * 实例的录制上下文
 * Created by guoyanfei .
 * 2019-01-23 .
 */
@Slf4j
public final class RecordContext {

    private static final int TRACE_QUEUE_SIZE = 500;

    /**
     * 用例集id
     */
    private final Long usecaseSetId;

    /**
     * 录制文件最大分钟数，单位分钟
     */
    private final Integer maxMinutes;

    private volatile Long startTime;

    private volatile Long endTime;

    /**
     * 录制文件最大大小，单位M
     */
    private final Integer maxFileSize;

    /**
     * 录制文件当前大小，单位byte
     */
    private volatile long currentFileSize;

    /**
     * trace的队列，每个请求录制完成往traceQueue中offer一条，另外有一个线程逐个poll traceQueue中的内容写文件
     */
    private LinkedBlockingQueue<FlowReplayTrace> traceQueue = new LinkedBlockingQueue<>(TRACE_QUEUE_SIZE);

    /**
     * 录制结束(录制文件还没写入完成)标记
     */
    private volatile boolean recordFinished = false;

    /**
     * 录制结果
     */
    private volatile RecordResult recordResult;

    /**
     * 构建录制详情所需的一些上下文内容
     * 目前只有堆栈内容
     */
    private final RecordDetailContext recordDetailContext;

    public RecordContext(RecordConfigDto recordConfig) {
        if (recordConfig == null || !recordConfig.isLegal()) {
            throw new FlowReplayException("录制配置为空or录制参数不合法");
        }
        this.usecaseSetId = recordConfig.getUsecaseSetId();

        this.maxMinutes = recordConfig.getMaxMinutes();
        this.startTime = null;
        this.endTime = null;

        this.maxFileSize = recordConfig.getMaxFileSize();
        this.currentFileSize = 0L;

        this.recordResult = null;

        this.recordDetailContext = new RecordDetailContext(recordConfig.getStackFramesPrefixWhitelist());
    }

    public RecordDetailContext getRecordDetailContext() {
        return recordDetailContext;
    }

    /**
     * 判断时间是否已经录制到需要结束的时候
     * @return
     */
    public boolean isTimeToEnd() {
        if (this.startTime == null || this.maxMinutes == null) {
            log.error("录制开始时间为空or最大录制分钟数为空_返回需要结束录制");
            return true;
        }
        long expectEndTime = startTime + maxMinutes * 60 * 1000;
        return expectEndTime < System.currentTimeMillis();
    }


    /**
     * 开始录制
     */
    public void start() {
        this.startTime = System.currentTimeMillis();
    }

    /**
     * 结束录制
     */
    public void end() {
        endTime = System.currentTimeMillis();
    }

    public void setRecordResult(RecordResult recordResult) {
        this.recordResult = recordResult;
    }

    public Long getUsecaseSetId() {
        return usecaseSetId;
    }

    public Integer getMaxFileSize() {
        return maxFileSize;
    }

    public Long getStartTime() {
        return startTime;
    }

    public Long getEndTime() {
        return endTime;
    }

    public long getCurrentFileSize() {
        return currentFileSize;
    }

    public RecordResult getRecordResult() {
        return recordResult;
    }

    public void setCurrentFileSize(long currentFileSize) {
        this.currentFileSize = currentFileSize;
    }

    public void offerTrace(FlowReplayTrace trace) {
        this.traceQueue.offer(trace);
    }

    public FlowReplayTrace pollTrace() throws InterruptedException {
        return this.traceQueue.poll(2, TimeUnit.SECONDS);
    }

    public boolean isTraceQueueEmpty() {
        return this.traceQueue.size() < 1;
    }

    public void recordFinished() {
        this.recordFinished = true;
    }

    public boolean isRecordFinished() {
        return this.recordFinished;
    }
}
