package cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.aop;

import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayConstants;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayErrorMsgTypeEnum;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.StepDiffColumn;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.*;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.PluginException;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.InstanceMethodsAroundInterceptor;
import cn.com.duibaboot.ext.autoconfigure.javaagent.core.interceptor.enhance.MethodInterceptResult;
import com.esotericsoftware.kryo.KryoException;
import lombok.extern.slf4j.Slf4j;

import java.lang.reflect.Method;
import java.util.Set;

/**
 * guava缓存的方法回归
 * Created by guoyanfei .
 * 2019-04-15 .
 */
@Slf4j
public class ReplayGuavaCacheMethodInterceptor implements InstanceMethodsAroundInterceptor {

    /**
     * 可以回归
     * @param method
     * @param allArguments
     * @return
     */
    private boolean canReplay(Method method, Object[] allArguments) {
        // 当前不是回归的环境 或者 当前请求不是回归请求
        if (!FlowReplayUtils.isReplayEnv() || !ReplayTraceContext.isReplaying()) {
            return false;
        }
        if (!FlowReplayConstants.GUAVA_CACHE_AOP_METHOD.contains(method.getName())) {
            return false;
        }
        if (FlowReplayUtils.isGlobalWhitelist(allArguments)) {
            return false;
        }
        return true;
    }

    @Override
    public void beforeMethod(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, MethodInterceptResult result) throws Throwable {
        if (!canReplay(method, allArguments)) {
            return;
        }

        FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

        // 创建回归的span，用于回归详情的构建
        GuavaCacheFlowReplaySpan replayDetailSpan = createReplayDetailSpan(method, allArguments, argumentsTypes, span);

        // 增加了新的调用 || 调用内容有变动
        if (span == null || SpanType.GUAVA_CACHE != span.getSpanType()) {
            String expert = SpanType.GUAVA_CACHE.name();
            String actual = span != null ? span.getSpanType().name() : null;
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        GuavaCacheFlowReplaySpan guavaSpan = (GuavaCacheFlowReplaySpan) span;

        if (!FlowReplayUtils.isMethodEqual(guavaSpan.getMethodName(), method)) {
            String expert = guavaSpan.getMethodName();
            String actual = method.getName();
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_402, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        if (!FlowReplayUtils.isArgumentsTypesEqual(guavaSpan.getParameterTypes(), GuavaCacheFlowReplaySpan.customizeArgumentsTypes(method, argumentsTypes))) {
            String expert = FlowReplayUtils.stringArrayToString(guavaSpan.getParameterTypes());
            String actual = FlowReplayUtils.classArrayToString(argumentsTypes);
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_400, expert, actual);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            // 对比 [录制的参数列表] 和 [回归的参数列表] ，记录对比过程中不相同的参数
            Object[] recordParams = guavaSpan.getParameterValues();
            Object[] replayParams = GuavaCacheFlowReplaySpan.customizeParameterValues(method, allArguments);
            Set<StepDiffColumn> stepDiffColumns = FlowReplayUtils.compareArray(ReplayTraceContext.getContextTraceId(), ReplayTraceContext.getCurrentSpanIdx(), recordParams, replayParams);
            ReplayTraceContext.addAllStepDiffColumns(stepDiffColumns);
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_403, e);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }

        try {
            Object mockRet = guavaSpan.getReturnValue();
            // 参数都相同的情况下，mock之前把mock的返回值回填到构建回归详情用的span中
            replayDetailSpan.setRet(method, mockRet);
            result.defineReturnValue(mockRet);
        } catch (KryoException e) {
            ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_404, e);
            throw new PluginException(ReplayTraceContext.getCompletedErrorMsg());
        }
    }

    private GuavaCacheFlowReplaySpan createReplayDetailSpan(Method method, Object[] allArguments, Class<?>[] argumentsTypes, FlowReplaySpan span) {
        GuavaCacheFlowReplaySpan replayDetailSpan = GuavaCacheFlowReplaySpan.createSpan(method, allArguments, argumentsTypes, null);
        replayDetailSpan.setTraceId(FlowReplayTrace.getCurrentTraceId());
        if (span != null) {
            replayDetailSpan.setSpanId(span.getSpanId());
        }
        FlowReplayTrace.addSubSpan(replayDetailSpan);
        return replayDetailSpan;
    }

    @Override
    public Object afterMethod(Object zuperCall, Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Object ret) throws Throwable {
        return ret;
    }

    @Override
    public void handleMethodException(Object obj, Method method, Object[] allArguments, Class<?>[] argumentsTypes, Throwable t) {
        //do nothing
    }
}
