package cn.com.duibaboot.ext.autoconfigure.rocketmq;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.*;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.client.producer.DefaultMQProducer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.boot.web.context.WebServerInitializedEvent;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;

import java.util.Map;

/**
 * RocketMQ生产者/消费者自动配置
 * Created by wenqi.huang on 2017/04/18.
 */
@Configuration
@ConditionalOnClass({DefaultMQProducer.class})
@EnableConfigurationProperties(RocketMqProperties.class)
public class RocketMqAutoConfiguration {
    private static final Logger logger = LoggerFactory.getLogger(RocketMqAutoConfiguration.class);

    protected abstract static class RocketMqClientConfiguration{

        @Autowired
        protected RocketMqProperties rocketMqProperties;

    }

    @Configuration
    @ConditionalOnClass({DefaultMQProducer.class})
    @ConditionalOnMissingBean(name={"rocketMqProducer"})
    @ConditionalOnProperty(name="duiba.rocketmq.producer.enable", havingValue = "true", matchIfMissing = false)
    protected static class RocketMqProducerConfigurator extends RocketMqClientConfiguration{

        @Bean(name="rocketMqProducer", destroyMethod = "shutdown")
        public DefaultMQProducer rocketMqProducer() throws MQClientException {
            /**
             * 一个应用创建一个Producer，由应用来维护此对象，可以设置为全局对象或者单例<br>
             */
            DefaultMQProducerWrapper p = new DefaultMQProducerWrapper();

            /**
             * producerGroup 这个概念发送普通的消息时，作用不大，但是发送分布式事务消息时，比较关键，
             * 因为服务器会回查这个Group下的任意一个Producer
             * 建议把producerGroup设置成应用名
             */
            p.setProducerGroup(rocketMqProperties.getProducer().getGroup());
            p.setNamesrvAddr(rocketMqProperties.getNameSrvAddr());
            //p.setInstanceName("Producer");
            p.setSendMsgTimeout(rocketMqProperties.getProducer().getSendMsgTimeoutMillis());//单位：ms
            /**
             * Producer对象在使用之前必须要调用start初始化，初始化一次即可<br>
             * 注意：切记不可以在每次发送消息时，都调用start方法
             */
            p.start();

            return p;
        }
    }

    @Configuration
    @ConditionalOnClass({DefaultMQPushConsumer.class})
    protected static class RocketMqConsumerConfigurator extends RocketMqClientConfiguration{

        /**
         * 对MessageListener进行包装，增加cat监控和DBTimeProfile监控
         * @return
         */
        @Bean
        public static SpecifiedBeanPostProcessor<MessageListener> rocketmqMessageListenerPostProcessor(){
            return new RocketmqMessageListenerPostProcessor();
        }

        @Bean(name="rocketMqConsumer")
        @ConditionalOnMissingBean(name={"rocketMqConsumer"})
        @ConditionalOnProperty(name="duiba.rocketmq.consumer.enable", havingValue = "true", matchIfMissing = false)
        public DefaultMQPushConsumerWrapper rocketMqConsumer() {
            return new DefaultMQPushConsumerWrapper(rocketMqProperties, -1);
        }

        @Bean(name="extraRocketMqConsumer0")
        @ConditionalOnMissingBean(name={"extraRocketMqConsumer0"})
        @ConditionalOnProperty(name="duiba.rocketmq.extra-consumer[0].enable", havingValue = "true", matchIfMissing = false)
        public DefaultMQPushConsumerWrapper extraRocketMqConsumer0() {
            return new DefaultMQPushConsumerWrapper(rocketMqProperties, 0);
        }

        @Bean(name="extraRocketMqConsumer1")
        @ConditionalOnMissingBean(name={"extraRocketMqConsumer1"})
        @ConditionalOnProperty(name="duiba.rocketmq.extra-consumer[1].enable", havingValue = "true", matchIfMissing = false)
        public DefaultMQPushConsumerWrapper extraRocketMqConsumer1() {
            return new DefaultMQPushConsumerWrapper(rocketMqProperties, 1);
        }

        @Bean(name="extraRocketMqConsumer2")
        @ConditionalOnMissingBean(name={"extraRocketMqConsumer2"})
        @ConditionalOnProperty(name="duiba.rocketmq.extra-consumer[2].enable", havingValue = "true", matchIfMissing = false)
        public DefaultMQPushConsumerWrapper extraRocketMqConsumer2() {
            return new DefaultMQPushConsumerWrapper(rocketMqProperties, 2);
        }

        @EventListener(MainContextRefreshedEvent.class)
        @Order(Ordered.LOWEST_PRECEDENCE)//用户应用可能要加载缓存
        public void onEvent(MainContextRefreshedEvent event){
            if(FlowReplayUtils.isReplayEnv()){//流量回归实例不开启消费
                return;
            }
            Map<String, DefaultMQPushConsumerWrapper> map = event.getApplicationContext().getBeansOfType(DefaultMQPushConsumerWrapper.class);
            if(!map.isEmpty()){
                for(DefaultMQPushConsumerWrapper consumer : map.values()){
                    try {
                        //Consumer对象在使用之前必须要调用start初始化，初始化一次即可
                        if(consumer != null) {
                            consumer.startRun();
                        }
                    } catch (MQClientException e) {
                        throw new RuntimeException(e);
                    }
                }
            }
        }

    }
}
