package cn.com.duibaboot.ext.autoconfigure.web;

import cn.com.duibaboot.ext.autoconfigure.core.SpecifiedBeanPostProcessor;
import cn.com.duibaboot.ext.autoconfigure.web.mvc.*;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.BeanFactory;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnWebApplication;
import org.springframework.boot.autoconfigure.http.HttpMessageConverters;
import org.springframework.boot.autoconfigure.web.servlet.WebMvcAutoConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.web.accept.ContentNegotiationManager;
import org.springframework.web.servlet.view.ContentNegotiatingViewResolver;

import java.util.Collections;
import java.util.List;

/**
 * 把ContentNegotiatingViewResolver配置为最低顺序，以防mvcAutoConfiguration自动注入最高优先级的ContentNegotiatingViewResolver
 * <br/>
 * 这样做是为了防止线上出现大量html.vm/jpg.vm/xml.vm找不到的问题。
 * 此问题重现的一种方法是在访问时先加入.html后缀，然后用正常链接访问。
 * <br/>
 * ContentNegotiatingViewResolver的大概目的是自动判断当前访问的页面，然后自动附加html等后缀寻找对应文件，我们不需要该特性
 * Created by wenqi.huang on 2017/1/5.
 */
@Configuration
@AutoConfigureBefore({WebMvcAutoConfiguration.class})
@ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
public class WebMvcFixAutoConfiguration {

    /**
     * 把ContentNegotiatingViewResolver配置为最低顺序，以防mvcAutoConfiguration自动注入最高优先级的ContentNegotiatingViewResolver
     * <br/>
     * 这样做是为了防止线上出现大量html.vm/jpg.vm/xml.vm找不到的问题。
     * 此问题重现的一种方法是在访问时先加入.html后缀，然后用正常链接访问。
     * <br/>
     * ContentNegotiatingViewResolver的大概目的是自动判断当前访问的页面，然后自动附加html等后缀寻找对应文件，我们不需要该特性
     */
    @Bean(name = "viewResolver")
    @ConditionalOnWebApplication(type = ConditionalOnWebApplication.Type.SERVLET)
    @ConditionalOnMissingBean(name = "viewResolver", value = ContentNegotiatingViewResolver.class)
    public ContentNegotiatingViewResolver viewResolver(BeanFactory beanFactory) {
        ContentNegotiatingViewResolver resolver = new ContentNegotiatingViewResolver();
        resolver.setContentNegotiationManager(
                beanFactory.getBean(ContentNegotiationManager.class));
        // ContentNegotiatingViewResolver uses all the other view resolvers to locate
        // a view so it should have a high precedence
        resolver.setOrder(Ordered.LOWEST_PRECEDENCE);
        return resolver;
    }

    /**
     * 这个HandlerExceptionResolver用于对BizException类进行特殊处理，返回的错误json字符串中多加入一个code字段
     * @return
     */
    @Bean
    public BizExceptionResolver bizExceptionResolver(){
        return new BizExceptionResolver();
    }

    @Bean
    public CustomWebMvcRegistrations customWebMvcRegistrations(){
        return new CustomWebMvcRegistrations();
    }

    /**
     * 把HttpMessageConvertersAutoConfiguration中定义的HttpMessageConverters替换为使用自定义的HttpMessageConverters，以在最后增加fst和kryo的HttpMessageConverter.(顺序很重要，一定要在最后)
     * @return
     */
    @Bean
    public static SpecifiedBeanPostProcessor httpMessageConvertersPostProcessor(){
        return new SpecifiedBeanPostProcessor<HttpMessageConverters>(){

            @Override
            public int getOrder() {
                return 0;
            }

            @Override
            public Class<HttpMessageConverters> getBeanType() {
                return HttpMessageConverters.class;
            }

            @Override
            public Object postProcessBeforeInitialization(HttpMessageConverters bean, String beanName) throws BeansException {
                return bean;
            }

            @Override
            public Object postProcessAfterInitialization(HttpMessageConverters bean, String beanName) throws BeansException {
                List<HttpMessageConverter<?>> converters = bean.getConverters();
                return new CustomHttpMessageConverters(false, converters == null
                        ? Collections.emptyList() : converters);
            }
        };
    }
}
