package org.springframework.cloud.openfeign;

import cn.com.duiba.boot.cat.CatWithArgs;
import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.catmonitor.constants.CatConstants;
import cn.com.duibaboot.ext.autoconfigure.cat.context.CatContext;
import cn.com.duibaboot.ext.autoconfigure.core.rpc.RpcContext;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayErrorMsgTypeEnum;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayException;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.FlowReplayUtils;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.record.aop.IgnoreSubInvokesContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.ReplayTraceContext;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.replay.StepDiffColumn;
import cn.com.duibaboot.ext.autoconfigure.flowreplay.span.*;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Event;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import com.dianping.cat.message.internal.AbstractMessage;
import com.esotericsoftware.kryo.KryoException;
import com.google.common.annotations.VisibleForTesting;
import com.netflix.hystrix.exception.HystrixBadRequestException;
import lombok.extern.slf4j.Slf4j;
import org.aopalliance.intercept.MethodInterceptor;

import java.lang.reflect.InvocationHandler;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Proxy;
import java.util.Objects;
import java.util.Set;

/**
 * 拦截FeignClient的调用加入cat监控功能，以及拦截并注入当前调用的方法
 */
@Slf4j
public class CustomFeignClientFactoryBean extends FeignClientFactoryBean {

    //客户端应用名，CustomFeignClientsRegistrar.customFeignClientBeanDefinition会注入
//    @Value("${spring.application.name}")
    private String springApplicationName;

    @Override
    public Object getObject() throws Exception {
        //增加cat监控功能
        Class classType = super.getObjectType();
        Object feignClient = super.getObject();

        Object proxyInstance = proxyInstance(classType, feignClient);
        if (FlowReplayUtils.isReplayEnv()) {
            proxyInstance = Proxy.newProxyInstance(proxyInstance.getClass().getClassLoader(), new Class[] { classType }, new FlowReplayFeignClientInvocationHandler(proxyInstance));
        } else {
            proxyInstance = Proxy.newProxyInstance(proxyInstance.getClass().getClassLoader(), new Class[] { classType }, new FlowRecordFeignClientInvocationHandler(proxyInstance));
        }
        return proxyInstance;
    }

    private Object proxyInstance(Class classType, Object feignClient) {
        if (CatUtils.isCatClassExists()) {
            return Proxy.newProxyInstance(classType.getClassLoader(), new Class[] { classType }, new FeignClientInvocationHandlerWithCat(springApplicationName, feignClient, this.getName()));
        } else {
            return Proxy.newProxyInstance(classType.getClassLoader(), new Class[] { classType }, new FeignClientInvocationHandler(springApplicationName, feignClient, this.getName()));
        }
    }

    public static abstract class FlowReplayInvocationHandler implements InvocationHandler {

        protected final Object original;

        FlowReplayInvocationHandler(Object original) {
            this.original = original;
        }

        protected Object invokeMethodInner(Method method, Object target, Object[] args) throws Throwable {
            try {
                return method.invoke(target, args);
            } catch (Throwable e) {
                if (e instanceof InvocationTargetException) {
                    e = ((InvocationTargetException) e).getTargetException();
                }
                //通常这步拿出来的异常只会有以下2种：HystrixBadRequestException, FeignClientException
                if (e instanceof HystrixBadRequestException && e.getCause() != null && e.getCause() instanceof BizException) {
                    e = e.getCause();//如果cause是BizException，则应该把BizException提取出来抛出
                }
                throw e;
            }
        }
    }

    public static class FlowRecordFeignClientInvocationHandler extends FlowReplayInvocationHandler {

        FlowRecordFeignClientInvocationHandler(Object original) {
            super(original);
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            String methodName = method.getName();

            if ("equals".equals(methodName)
                || "hashCode".equals(methodName)
                || "toString".equals(methodName)
                || method.isDefault()
                || !FlowReplayTrace.isTraced()
                || IgnoreSubInvokesContext.isMarked()) {
                return invokeMethodInner(method, original, args);
            }

            IgnoreSubInvokesContext.instMark(original, method.getName(), args);

            Object ret;
            try {
                ret = invokeMethodInner(method, original, args);
            } catch (Throwable t) {
                // 如果捕捉到异常，并且正在录制中，那么不录这个用例
                if (FlowReplayTrace.isTraced()) {
                    FlowReplayTrace.remove();
                    IgnoreSubInvokesContext.unmark();
                }
                throw t;
            }
            try {
                FeignClientFlowReplaySpan span = FeignClientFlowReplaySpan.createSpan(method, args, ret);
                span.setTraceId(FlowReplayTrace.getCurrentTraceId());
                FlowReplayTrace.addSubSpan(span);
            } catch (Throwable t) {
                log.error("feignClient录制异常", t);
                // 录制异常，这个用例不录了
                FlowReplayTrace.remove();
            } finally {
                IgnoreSubInvokesContext.unmark();
            }
            return ret;
        }
    }

    public static class FlowReplayFeignClientInvocationHandler extends FlowReplayInvocationHandler {

        FlowReplayFeignClientInvocationHandler(Object original) {
            super(original);
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable { //NOSONAR
            String methodName = method.getName();

            if ("equals".equals(methodName)
                || "hashCode".equals(methodName)
                || "toString".equals(methodName)
                || method.isDefault()) {
                return invokeMethodInner(method, original, args);
            }

            // 本机是回归环境 并且 当前请求是回归请求 并且当前请求不是子调用
            if (FlowReplayUtils.isReplayEnv() && ReplayTraceContext.isReplaying()) {
                FlowReplaySpan span = ReplayTraceContext.pollSubSpan();

                FeignClientFlowReplaySpan replayDetailSpan = createReplayDetailSpan(method, args, span);

                // 增加了新的调用 || 调用内容有变动
                if (span == null || SpanType.FEIGN_CLIENT != span.getSpanType()) {
                    String expert = SpanType.FEIGN_CLIENT.name();
                    String actual = span != null ? span.getSpanType().name() : null;
                    ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_001, expert, actual);
                    throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
                }

                FeignClientFlowReplaySpan feignClientSpan = (FeignClientFlowReplaySpan) span;

                // 判断录制的span的调用和本次调用是否一致，如果一致，那么直接mock录制的span的返回值
                if (isRequestEqual(feignClientSpan, method, args)) {
                    try {
                        // 都相同mock返回值
                        Object mockRet = feignClientSpan.getReturnValue();

                        // 参数都相同的情况下，mock之前把mock的返回值回填到构建回归详情用的span中
                        replayDetailSpan.setReturnValue(mockRet);

                        return mockRet;
                    } catch (KryoException e) {
                        ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_207, e);
                    }
                }
                throw new FlowReplayException(ReplayTraceContext.getCompletedErrorMsg());
            } else {
                return invokeMethodInner(method, original, args);
            }
        }

        private FeignClientFlowReplaySpan createReplayDetailSpan(Method method, Object[] args, FlowReplaySpan span) {
            FeignClientFlowReplaySpan replayDetailSpan = FeignClientFlowReplaySpan.createSpan(method, args, null);
            replayDetailSpan.setTraceId(FlowReplayTrace.getCurrentTraceId());
            if (span != null) {
                replayDetailSpan.setSpanId(span.getSpanId());
            }
            FlowReplayTrace.addSubSpan(replayDetailSpan);
            return replayDetailSpan;
        }

        /**
         * 判断录制的span的调用和本次调用是否一致
         * 需要判断以下几点：
         * 1、类的全路径
         * 2、方法名称
         * 3、参数列表
         * 4、参数值列表
         * 5、返回值类型
         * @param feignClientSpan
         * @param currentMethod
         * @param currentArgs
         * @return
         */
        private boolean isRequestEqual(FeignClientFlowReplaySpan feignClientSpan, Method currentMethod, Object[] currentArgs) {
            if (!isTypeFullPathEqual(feignClientSpan, currentMethod)) {
                String expert = feignClientSpan.getTypeFullPath();
                String actual = currentMethod.getDeclaringClass().getName();
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_205, expert, actual);
                return false;
            }
            if (!isApiNameEqual(feignClientSpan, currentMethod)) {
                String expert = feignClientSpan.getMethodName();
                String actual = currentMethod.getName();
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_201, expert, actual);
                return false;
            }
            if (!isParameterTypesEqual(feignClientSpan, currentMethod)) {
                String expert = FlowReplayUtils.stringArrayToString(feignClientSpan.getParameterTypes());
                String actual = FlowReplayUtils.classArrayToString(currentMethod.getParameterTypes());
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_202, expert, actual);
                return false;
            }
            try {
                // 对比 [录制的参数列表] 和 [回归的参数列表] ，记录对比过程中不相同的参数
                Set<StepDiffColumn> stepDiffColumns = FlowReplayUtils.compareArray(ReplayTraceContext.getContextTraceId(), ReplayTraceContext.getCurrentSpanIdx(), feignClientSpan.getParameterValues(), currentArgs);
                ReplayTraceContext.addAllStepDiffColumns(stepDiffColumns);
            } catch (KryoException e) {
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_206, e);
                return false;
            }

            if (!isReturnTypeEqual(feignClientSpan, currentMethod)) {
                ReplayTraceContext.markError(FlowReplayErrorMsgTypeEnum.EM_204);
                return false;
            }

            return true;
        }

        private boolean isTypeFullPathEqual(FeignClientFlowReplaySpan feignClientSpan, Method currentMethod) {
            String spanTypeFullPath = feignClientSpan.getTypeFullPath();
            String currentTypeFullPath = currentMethod.getDeclaringClass().getName();
            return Objects.equals(spanTypeFullPath, currentTypeFullPath);
        }

        private boolean isApiNameEqual(FeignClientFlowReplaySpan feignClientSpan, Method currentMethod) {
            String spanApiName = feignClientSpan.getApiName();
            String currentApiName = FlowReplayUtils.parseApiNameByMethod(currentMethod);
            return Objects.equals(spanApiName, currentApiName);
        }

        private boolean isParameterTypesEqual(FeignClientFlowReplaySpan feignClientSpan, Method currentMethod) {
            String[] spanParameterTypes = feignClientSpan.getParameterTypes();
            Class[] currentParameterTypes = currentMethod.getParameterTypes();

            int currentLength = currentParameterTypes.length;
            String[] currentParameterTypeNames = null;
            if (currentLength > 0) {
                currentParameterTypeNames = new String[currentLength];
                for (int i = 0; i < currentLength; i++) {
                    currentParameterTypeNames[i] = currentParameterTypes[i].getName();
                }
            }

            return Objects.deepEquals(spanParameterTypes, currentParameterTypeNames);
        }

        private boolean isReturnTypeEqual(FeignClientFlowReplaySpan feignClientSpan, Method currentMethod) {
            Object spanReturnType = feignClientSpan.getReturnType();
            return Objects.equals(spanReturnType, currentMethod.getReturnType().getName());
        }
    }

    public void setSpringApplicationName(String springApplicationName) {
        this.springApplicationName = springApplicationName;
    }

    /**
     * {@link MethodInterceptor}
     */
    @VisibleForTesting
    public static class FeignClientInvocationHandler implements InvocationHandler {

        protected final String springApplicationName;
        protected final String serverName;
        protected final Object original;

        FeignClientInvocationHandler(String springApplicationName, Object original, String serverName) {
            this.springApplicationName = springApplicationName;
            this.original = original;
            this.serverName = serverName;
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            String methodName = method.getName();

            if ("equals".equals(methodName)
                    || "hashCode".equals(methodName)
                    || "toString".equals(methodName)
                    || method.isDefault()) {
                return invokeMethodInner(method, original, args);
            }

            RpcContext.getContext().setMethod(method);
            RpcContext.getContext().setInvokeArgs(args);
            RpcContext.getContext().setSourceServiceId(springApplicationName);
            RpcContext.getContext().setTargetServiceId(serverName);

            return invokeMethodInner(method, original, args);
        }

        protected Object invokeMethodInner(Method method, Object target, Object[] args) throws Throwable{
            try {
                return method.invoke(target, args);
            } catch (Throwable e) {
                if(e instanceof InvocationTargetException){
                    e = ((InvocationTargetException)e).getTargetException();
                }
                //通常这步拿出来的异常只会有以下2种：HystrixBadRequestException, FeignClientException
                if(e instanceof HystrixBadRequestException && e.getCause() != null && e.getCause() instanceof BizException){
                    e = e.getCause();//如果cause是BizException，则应该把BizException提取出来抛出
                }
                throw e;
            }
        }

    }

    /**
     * {@link MethodInterceptor}
     */
    @VisibleForTesting
    public static class FeignClientInvocationHandlerWithCat extends FeignClientInvocationHandler {

        FeignClientInvocationHandlerWithCat(String springApplicationName, Object original, String serverName) {
            super(springApplicationName, original, serverName);
        }

        @Override
        public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
            String methodName = method.getName();

            if ("equals".equals(methodName)
                    || "hashCode".equals(methodName)
                    || "toString".equals(methodName)
                    || method.isDefault()) {
                return invokeMethodInner(method, original, args);
            }

            RpcContext.getContext().setMethod(method);
            RpcContext.getContext().setInvokeArgs(args);
            RpcContext.getContext().setSourceServiceId(springApplicationName);
            RpcContext.getContext().setTargetServiceId(serverName);

            if(!CatUtils.isCatEnabled()){
                return invokeMethodInner(method, original, args);
            }

            return invokeWithCatTransaction(method, args);
        }

        private Object invokeWithCatTransaction(Method method, Object[] args) throws Throwable{
            String loggerName = resolveLoggerName(method, args);

            Transaction transaction = Cat.newTransaction(CatConstants.CROSS_CONSUMER, loggerName);

            try {
                CatContext context = new CatContext();

                Cat.logRemoteCallClient(context);
                setAttachment(context, method);

                Object result = invokeMethodInner(method, original, args);

                transaction.setStatus(Message.SUCCESS);
                return result;
            }catch(Throwable e){
                if(!(e instanceof BizException)) {
                    Cat.logError(e);//BizException是正常的业务异常，比如用户不存在，不应该记录到cat.
                    transaction.setStatus(e);
                }else{
                    //记录BizException
                    Event bizExceptionEvent = Cat.newEvent("BizException", String.format("BizException(message:%s,code:%s)(this exception will not trigger hystrix)",e.getMessage(),((BizException) e).getCode()));
                    completeEvent(bizExceptionEvent);

                    transaction.addChild(bizExceptionEvent);

                    transaction.setStatus(Message.SUCCESS);
                }
                throw e;
            }finally {
                Event crossAppEvent =  Cat.newEvent(CatConstants.CONSUMER_CALL_APP, springApplicationName);
                Event crossServerEvent = Cat.newEvent(CatConstants.CONSUMER_CALL_SERVER, serverName);
                completeEvent(crossAppEvent);
                completeEvent(crossServerEvent);

                transaction.addChild(crossAppEvent);
                transaction.addChild(crossServerEvent);

                CatUtils.completeTransaction(transaction);
            }
        }

        private void completeEvent(Event event){
            AbstractMessage message = (AbstractMessage) event;
            message.setStatus(Event.SUCCESS);
            message.setCompleted(true);
        }

        private String resolveLoggerName(Method method, Object[] args){//NOSONAR
            StringBuilder loggerName = new StringBuilder(method.getDeclaringClass().getSimpleName())
                    .append(".").append(method.getName());

            //注解了CatWithArgs表示cat中的监控名需要附加参数，详情见CatWithArgs类的注释
            if(method.isAnnotationPresent(CatWithArgs.class)){
                CatWithArgs catWithArgs = method.getAnnotation(CatWithArgs.class);
                int[] argIndexes = catWithArgs.argIndexes();
                if(argIndexes == null || argIndexes.length == 0
                        || args == null || args.length == 0){
                    return loggerName.toString();
                }

                StringBuilder sb = new StringBuilder();
                for(int argIdx : argIndexes){
                    if(argIdx >= 0 && args.length > argIdx) {
                        Object obj = args[argIdx];
                        sb.append(obj == null ? "" : obj.toString()).append(",");
                    }
                }

                if(sb.length() > 0){
                    sb.deleteCharAt(sb.length() - 1);
                    sb.insert(0, "(");
                    sb.append(")");
                }
                loggerName.append(sb);
            }
            return loggerName.toString();
        }

        private void setAttachment(CatContext context, Method method){
            RpcContext.getContext().setAttachment(Cat.Context.ROOT,context.getProperty(Cat.Context.ROOT));
            RpcContext.getContext().setAttachment(Cat.Context.CHILD,context.getProperty(Cat.Context.CHILD));
            RpcContext.getContext().setAttachment(Cat.Context.PARENT,context.getProperty(Cat.Context.PARENT));
            RpcContext.getContext().setAttachment(cn.com.duibaboot.ext.autoconfigure.core.Constants.X_RPC_CLIENT, springApplicationName);
        }

    }

}
