package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor;
import org.springframework.cloud.context.scope.GenericScope;

/**
 * eurekaClient实例在 @ConditionalOnRefreshScope 情况下每次 refresh，会销毁重新创建。实例会取消注册，再重新注册到eureka。导致当前应用refresh时，其他应用从eureka获取该应用实例的时候存在找不到的情况
 * 此处扩展后，refresh不再会重新创建eurekaClient实例
 * @author wenqi.huang
 * @auther gyf
 * 2018/4/27 .
 */
public class EurekaClientBeanDefinitionRegistryPostProcessor implements BeanDefinitionRegistryPostProcessor {

    private static final Logger logger = LoggerFactory.getLogger(EurekaClientBeanDefinitionRegistryPostProcessor.class);

    private static final String EUREKA_CLIENT_BEAN_ID = "eurekaClient";

    @Override
    public void postProcessBeanDefinitionRegistry(BeanDefinitionRegistry registry) throws BeansException {
        BeanDefinition eurekaClientBeanDefinition = null;
        BeanDefinition scopedEurekaClientBeanDefinition = null;
        try {
            eurekaClientBeanDefinition = registry.getBeanDefinition(EUREKA_CLIENT_BEAN_ID);
            //如果eurekaClient是@RefreshScope的，则一定会有scopedTarget开头的bean
            scopedEurekaClientBeanDefinition = registry.getBeanDefinition(GenericScope.SCOPED_TARGET_PREFIX + EUREKA_CLIENT_BEAN_ID);
        } catch (NoSuchBeanDefinitionException e) {
            //Ignore
            return;
        }

        if (eurekaClientBeanDefinition == null || scopedEurekaClientBeanDefinition == null) {
            return;
        }

        BeanDefinition eurekaClientOriginatingBeanDefinition = eurekaClientBeanDefinition.getOriginatingBeanDefinition();
        if (eurekaClientOriginatingBeanDefinition == null) {
            throw new IllegalStateException("[NOTIFYME]如果你看到这个异常，说明我们不支持spring-cloud或spring-boot的新版本，请联系中间件团队解决此问题");
        }

        registry.removeBeanDefinition(GenericScope.SCOPED_TARGET_PREFIX + EUREKA_CLIENT_BEAN_ID);

        //spring内部生成代理beanDefinition时会把原始的beanDefinition的AutowireCandidate设为false，导致无法注入到其他bean中，这里把这个配置还原回去
        eurekaClientOriginatingBeanDefinition.setAutowireCandidate(true);
        eurekaClientOriginatingBeanDefinition.setScope(BeanDefinition.SCOPE_SINGLETON);//把scope由refresh改为singleton

        registry.removeBeanDefinition(EUREKA_CLIENT_BEAN_ID);
        registry.registerBeanDefinition(EUREKA_CLIENT_BEAN_ID, eurekaClientOriginatingBeanDefinition);
    }

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        //do nothing
    }

}
