package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign;

import com.caucho.hessian.io.Hessian2Input;
import com.caucho.hessian.io.Hessian2Output;
import com.esotericsoftware.kryo.Kryo;
import com.esotericsoftware.kryo.io.Input;
import com.esotericsoftware.kryo.io.Output;
import org.apache.commons.lang3.StringUtils;
import org.nustaq.serialization.FSTConfiguration;
import org.springframework.boot.context.properties.ConfigurationProperties;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;

@ConfigurationProperties(prefix = "duiba.feign")
public class DuibaFeignProperties {
    /**
     * spring-cloud-netflix调用使用的序列化方式，可选：hessian2/json/fst/kryo, 默认json(建议使用hessian2)
     * <br/>
     * 4种方式均支持：增加字段、删除字段。
     * 其中hessian2兼容性最好，允许任意改变字段类型，hessian2会尽量帮你转换，转换不了才报错，比如int/long/String可以互相改变字段类型;
     * fst不支持改变字段类型，即使是int/long之间的互相改变也不行;
     * kryo只支持简单的int/long字段类型互转，int和String无法互转,kryo还要求实体类必须有无参构造函数(比如不能使用Arrays.asList生成的List);
     * 经过测试，序列化后的byte[]大小为：json >> fst > kryo > hessian2;
     * 经过测试，序列化/反序列化时间差距不是很大.
     * <br/>
     * 二进制序列化方式都基于Field，json序列化方式一般都基于get方法。
     */
    private String serialization = "json";

    public String getSerialization() {
        return serialization;
    }

    public DuibaFeignSerialization getSerializationEnum(){
        return DuibaFeignSerialization.typeOf(serialization);
    }

    public void setSerialization(String serialization) {
        this.serialization = serialization;
    }

    /**
     * 当需要增加一个新的序列化方式时，只需要在这里增加新的枚举即可。
     */
    public enum DuibaFeignSerialization{
        FST("fst", "fst-bytes/fst"){
            @Override
            public byte[] serialize(Object obj) {
                if(obj == null){
                    return null;
                }
                return fstConfiguration.asByteArray(obj);
            }

            @Override
            public <T> T deserialize(byte[] bs) {
                if(bs == null || bs.length == 0){
                    return null;
                }
                return (T)fstConfiguration.asObject(bs);
            }
        },
        KRYO("kryo", "kryo-bytes/kryo") {
            @Override
            public byte[] serialize(Object obj) {
                if(obj == null){
                    return null;
                }
                ByteArrayOutputStream out = new ByteArrayOutputStream();
                Output output = new Output(out);
                kryo.writeClassAndObject(output, obj);
                output.close();
                return out.toByteArray();
            }

            @Override
            public <T> T deserialize(byte[] bs) {
                if(bs == null || bs.length == 0){
                    return null;
                }
                ByteArrayInputStream in = new ByteArrayInputStream(bs);
                Input input = new Input(in);
                Object obj = kryo.readClassAndObject(input);
                input.close();
                return (T)obj;
            }
        },
        HESSIAN2("hessian2", "hessian2-bytes/hessian2") {
            @Override
            public byte[] serialize(Object obj) {
                if(obj == null){
                    return null;
                }
                ByteArrayOutputStream out = new ByteArrayOutputStream();
                Hessian2Output hessian2Output = new Hessian2Output(out);
                try {
                    hessian2Output.writeObject(obj);
                    hessian2Output.flush();
                }catch(IOException e){
                    throw new RuntimeException(e);
                }
                return out.toByteArray();
            }

            @Override
            public <T> T deserialize(byte[] bs) {
                if(bs == null || bs.length == 0){
                    return null;
                }
                ByteArrayInputStream in = new ByteArrayInputStream(bs);
                Hessian2Input hessian2Input = new Hessian2Input(in);
                try {
                    return (T)hessian2Input.readObject();
                } catch (IOException e) {
                    throw new RuntimeException(e);
                }
            }
        },
        JSON("json", "application/json") {
            @Override
            public byte[] serialize(Object obj) {
                throw new UnsupportedOperationException("not support json serialize, please serialize yourself");
            }

            @Override
            public Object deserialize(byte[] bs) {
                throw new UnsupportedOperationException("not support json deserialize, please deserialize yourself");
            }
        };

        private String type;
        private String contentType;

        private static FSTConfiguration fstConfiguration;
        private static Kryo kryo;

        static{
            try {
                fstConfiguration = FSTConfiguration.createDefaultConfiguration();
            }catch(Throwable e){
                //Ignore
            }
            try {
                kryo = new Kryo();
            }catch(Throwable e){
                //Ignore
            }
        }

        DuibaFeignSerialization(String type, String contentType){
            this.type = type;
            this.contentType = contentType;
        }

        public String getType(){
            return type;
        }
        public String getContentType(){
            return contentType;
        }

        public abstract byte[] serialize(Object obj);
        public abstract <T> T deserialize(byte[] bs);

        public static DuibaFeignSerialization typeOf(String type){
            for(DuibaFeignSerialization s : DuibaFeignSerialization.values()){
                if(s.type.equals(type)){
                    return s;
                }
            }

            throw new IllegalArgumentException("serialization type：" + type + " is illegal, must be one of: fst/kryo/json");
        }

        public static DuibaFeignSerialization contentTypeOf(String contentType){
            if(StringUtils.isEmpty(contentType)){
                return null;
            }
            for(DuibaFeignSerialization s : DuibaFeignSerialization.values()){
                if(s.contentType.equals(contentType)
                    || contentType.startsWith(s.contentType + ";")){//有可能是hession2-bytes/hessian2; charset=UTF-8这样的格式，做个兼容
                    return s;
                }
            }

            return null;
        }
    }
}
