package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon

import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter
import cn.com.duiba.boot.utils.JarVersionUtils
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataAutoConfiguration
import com.netflix.loadbalancer.Server
import com.netflix.niws.loadbalancer.DiscoveryEnabledServer
import org.apache.commons.lang.StringUtils

/**
 * ribbon列表过滤器,此类用于跟压测配合，用于隔离压测流量和正常流量，压测专用服务在注册到eureka中时会附带压测场景ID，当前服务在发起调用前需要判读如下逻辑：
 * <br/>
 * 1. 如果当前服务器带有场景ID（压测专用，不允许接受正常流量），则过滤eureka服务器列表时只会选择场景ID相同的目标服务器进行调用(如果找不到会抛出异常，防止调用到了正常实例)；
 * 2. 如果当前服务器没有带场景ID（正常流量专用，不允许接受压测流量），则过滤eureka服务器列表时只会选择没有场景ID的目标服务器进行调用；
 *
 * <br/>
 * 这个类不能加到压测包下，因为压测包下的类只有在引入perftest模块的情况下才会生效，而此类在没有引入perftst模块的情况下也应该生效，否则正常流量可能会调用到压测专用服务器上。
 */
data class PerfTestRibbonServerListFilter(var thisServerPerfTestSceneId : String): RibbonServerListFilter {
    override fun filter(serverList: List<Server>, key: Any?): List<Server> {
        if(serverList.isEmpty() || !serverList.stream().allMatch({ s -> s is DiscoveryEnabledServer })) {
            return serverList
        }

        var retList : MutableList<Server> = ArrayList()
        for (server in serverList){
            if(server is DiscoveryEnabledServer){
                var serverPerfSceneId = server.instanceInfo.metadata[DiscoveryMetadataAutoConfiguration.DUIBA_PERF_SCENE_ID]
                if(StringUtils.isBlank(thisServerPerfTestSceneId)
                        && StringUtils.isBlank(serverPerfSceneId)){
                    //本机不是压测专用服务器，而且目标服务器也不是，则添加到允许调用的备选服务器列表
                    retList.add(server)
                }else if(StringUtils.equals(thisServerPerfTestSceneId, serverPerfSceneId)) {
                    var duibaBootVersion = server.instanceInfo.metadata[DiscoveryMetadataAutoConfiguration.DUIBA_BOOT_VERSION]
                    //本机是压测专用服务器，并且与目标服务器的场景ID相同，并且ext版本大于等于1.2.234， 则添加到允许调用的备选服务器列表
                    if(JarVersionUtils.isJarVersionEqualOrGreaterThan(duibaBootVersion, "1.2.234") == true) {
                        retList.add(server)
                    }
                }
            }
        }

        //如果当前是压测实例，却找不到被调应用的对应压测实例，直接抛出异常
        if(retList.isEmpty() && !StringUtils.isBlank(thisServerPerfTestSceneId)){
            //获得依赖的应用名字
            var dependencyAppName = (serverList.get(0) as DiscoveryEnabledServer).instanceInfo.appName
            if(!"ZIPKIN-SERVER".equals(dependencyAppName)) {//zipkin-server还是要允许调用的
                throw IllegalStateException("在进行spring-cloud调用前检测到当前实例为压测专用实例，但是没有找到" + dependencyAppName + "的压测专用实例(要求ext>=1.2.234)，拒绝调用。（压测场景ID:" + thisServerPerfTestSceneId + "）")
            }
        }

        return if(retList.isEmpty()) serverList else retList
    }
}
