package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon;

import ch.qos.logback.classic.Level;
import ch.qos.logback.classic.LoggerContext;
import cn.com.duibaboot.ext.autoconfigure.core.utils.SpringBootUtils;
import com.netflix.appinfo.InstanceInfo;
import com.netflix.client.config.IClientConfig;
import com.netflix.loadbalancer.AbstractLoadBalancerPing;
import com.netflix.loadbalancer.BaseLoadBalancer;
import com.netflix.loadbalancer.Server;
import com.netflix.niws.loadbalancer.DiscoveryEnabledServer;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.net.MalformedURLException;

/**
 * 使用自定义的ping组件，每隔3秒访问服务器/monitor/check接口，如果不通或者返回不是OK，则标记为dead状态，去除对该服务器的流量（官方的只判断了从eureka取下来的状态是否UP，存在很大延迟.）
 *
 * "Ping" Discovery Client
 * i.e. we dont do a real "ping". We just assume that the server is up if Discovery Client says so
 * @author stonse
 *
 */
public class RibbonPing extends AbstractLoadBalancerPing {

    private static final Logger logger = LoggerFactory.getLogger(RibbonPing.class);

    static{
        if(SpringBootUtils.isJarInJarMode()) {//不是本地开发模式才强制设置RibbonPing为INFO日志级别
            LoggerContext loggerContext = (LoggerContext) LoggerFactory.getILoggerFactory();
            loggerContext.getLogger(RibbonPing.class).setLevel(Level.INFO);
        }
    }

    private static final CloseableHttpClient httpClient = HttpClientBuilder.create()
            .setDefaultRequestConfig(RequestConfig.custom().setConnectTimeout(150).setSocketTimeout(150).setConnectionRequestTimeout(100).build())
            .setMaxConnPerRoute(1)
            .setMaxConnTotal(1000)
//            .evictExpiredConnections()//开启后台线程定时清理失效的连接
            .setUserAgent("RibbonPing")
            .disableAutomaticRetries()//禁止重试
            .disableCookieManagement()
            .useSystemProperties()//for proxy
            .setKeepAliveStrategy(new DefaultConnectionKeepAliveStrategy(){
                @Override
                public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
                    long time = super.getKeepAliveDuration(response, context);
                    if(time == -1){
                        time = 30000;//链接最多空闲30秒
                    }
                    return time;
                }
            })
            .build();

    public RibbonPing() {
        //do nothing
    }

    @Override
    public boolean isAlive(Server server) {//NOSONAR
        if(server == null){
            return false;
        }
        if(!(server instanceof DiscoveryEnabledServer)){
            return true;
        }

        boolean isAlive = true;

        DiscoveryEnabledServer dServer = (DiscoveryEnabledServer)server;
        InstanceInfo instanceInfo = dServer.getInstanceInfo();
        String appName = "undefined";
        if (instanceInfo!=null){
            InstanceInfo.InstanceStatus status = instanceInfo.getStatus();
            if (status!=null){
                isAlive = status.equals(InstanceInfo.InstanceStatus.UP);
            }
            appName = instanceInfo.getAppName();
        }

        Throwable exception = null;
        String monitorCheckResult = null;
        if(isAlive) {
//                String healthCheckUrl = dServer.getInstanceInfo().getHealthCheckUrl();
            String monitorCheckUrl = dServer.getInstanceInfo().getHomePageUrl();
            if(StringUtils.isBlank(monitorCheckUrl)){//如果没有homePageUrl，则降级为自己拼装（某些python、nodejs等应用没有这个字段）
                monitorCheckUrl = "http://" + dServer.getInstanceInfo().getIPAddr() + ":" + dServer.getInstanceInfo().getPort() + "/";
            }
            if(!monitorCheckUrl.endsWith("/")){
                monitorCheckUrl = monitorCheckUrl + "/";
            }
            monitorCheckUrl = monitorCheckUrl + "monitor/check";
            HttpGet httpGet = new HttpGet(monitorCheckUrl);
            try {
                CloseableHttpResponse resp = httpClient.execute(httpGet);//不能调用resp.close()否则tcp长连接会关闭
                int statusCode = resp.getStatusLine().getStatusCode();
                monitorCheckResult = IOUtils.toString(resp.getEntity().getContent());//正常会拿到OK，
                resp.getEntity().getContent().close();//这行让httpClient归还连接到连接池
                if (statusCode == 404) {
                    logger.error("server: {}({}) has no http endpoint /monitor/check, please setup,get help from here: http://cf.dui88.com/pages/viewpage.action?pageId=5252873",
                            dServer.getInstanceInfo().getAppName(), dServer.getHostPort());
                    isAlive = true;
                } else if (statusCode >= 500) {
                    isAlive = false;
                }
            } catch (MalformedURLException e) {
                logger.error(e.getMessage(), e);
                isAlive = false;
            } catch (Throwable e) {//很久之前这里catch的是IOException，由于后来出现过一次问题，由于httpClient内部抛出了OutOfMemoryError，导致RibbonCustomAutoConfiguration第260行不会return，最终导致RibbonCustomAutoConfiguration第275行空指针。
//                    e.printStackTrace();
                isAlive = false;
                exception = e;
            }
        }

        if(server.isAlive() != isAlive){
            String statusStr = isAlive?"alive":"dead";
            if(exception != null){
                logger.info("[ping in ribbon] detect that {}[{}] is {}，exceptionClass:{}, errorMsg:{}", appName, server.getHostPort(), statusStr, exception.getClass().getName(), exception.getMessage());
            }else{
                logger.info("[ping in ribbon] detect that {}[{}] is {}, [/monitor/check] result:{}", appName, server.getHostPort(), statusStr, monitorCheckResult);
            }
        }
        //TODO 这里可能会有并发问题，官方Server.isAliveFlag字段上没有volatile属性
        //TODO 还有个问题，虽然设置为非alive，但是eureka的线程每隔30秒获取到serverList后，会把所有server设置为alive，这段时间可能会有问题?好像没有问题
        return isAlive;
    }

    @Override
    public void initWithNiwsConfig(IClientConfig clientConfig) {
        //do nothing
    }

    public void destroy(){
        try {
            httpClient.close();
        } catch (IOException e) {
            logger.error("", e);
        }
    }

}
