package cn.com.duibaboot.ext.autoconfigure.cloud.netflix.ribbon;

import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupContext;
import cn.com.duiba.boot.netflix.ribbon.RibbonServerListFilter;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DiscoveryMetadataAutoConfiguration;
import cn.com.duibaboot.ext.autoconfigure.grouping.ServiceGroupUtils;
import com.netflix.loadbalancer.Server;
import com.netflix.niws.loadbalancer.DiscoveryEnabledServer;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * ribbon列表过滤器，此类用于配合容器组的测试环境多场景并行测试需求，容器组起的一些服务在注册到eureka中时会福袋分组标识，当前服务在发起调用前需要判断如下逻辑：
 * <br/>
 * 1、如果请求的Cookie/Header中带上了 _duibaServiceGroupKey=bizKey-id 的分组标识，
 * 流量优先转发给具有相同分组标识的 如果没有 -> 转发给没有分组标识的服务 如果没有 -> 随便转发
 * 2、如果请求的Cookie/Header中没有带上 _duibaServiceGroupKey=bizKey-id 的分组标识，
 * 流量优先转发给没有分组标识的服务 如果没有 -> 随便转发
 * Created by guoyanfei .
 * 2018/11/7 .
 */
public class ServiceGroupRibbonServerListFilter implements RibbonServerListFilter {

    @Override
    public List<Server> filter(List<Server> serverList, Object loadBalancerKey) {
        if (CollectionUtils.isEmpty(serverList)) {
            return serverList;
        }
        for (Server s : serverList) {
            if (!(s instanceof DiscoveryEnabledServer)) {
                return serverList;
            }
        }

        //优先从loadBalancerKey中获取，如果取不到才才从ServiceGroupContext中取（reactive不应该用ServiceGroupContext）
        String currentServiceGroupKey = null;
        if (loadBalancerKey != null && loadBalancerKey instanceof Map) {
            Map<String, Object> loadBalancerInfo = (Map<String, Object>)loadBalancerKey;
            currentServiceGroupKey = (String)loadBalancerInfo.get(ServiceGroupUtils.DUIBA_SERVICE_GROUP_KEY);
        }

        if(StringUtils.isBlank(currentServiceGroupKey)) {
            currentServiceGroupKey = ServiceGroupContext.getCurrentGroupKey();
        }

        List<Server> havePriorityList = new ArrayList<>();  // 优先转发目标服务列表
        List<Server> noGroupKeyServerList = new ArrayList<>();  // 没有分组标识的服务
        for (Server server : serverList) {
            String serverGroupKey = ((DiscoveryEnabledServer) server).getInstanceInfo().getMetadata().get(DiscoveryMetadataAutoConfiguration.DUIBA_SERVICE_GROUP_KEY);

            if (this.groupKeyEquals(currentServiceGroupKey, serverGroupKey)) {
                havePriorityList.add(server);
            }
            if (StringUtils.isBlank(serverGroupKey)) {
                noGroupKeyServerList.add(server);
            }
        }

        // 流量优先转发给具有相同分组标识的
        if (!havePriorityList.isEmpty()) {
            return havePriorityList;
        }
        // 没有找到优先转发的目标服务列表。降级：请求带上了分组标识，并且存在没有分组标识的服务列表，则转发给没有分组标识的服务列表
        if (StringUtils.isNotBlank(currentServiceGroupKey) && !noGroupKeyServerList.isEmpty()) {
            return noGroupKeyServerList;
        }
        // 真不行，什么也找不到，随便转发
        return serverList;
    }

    private boolean groupKeyEquals(String requestGroupKey, String serverGroupKey) {
        if (StringUtils.isBlank(requestGroupKey)) {
            requestGroupKey = StringUtils.EMPTY;
        }
        if (StringUtils.isBlank(serverGroupKey)) {
            serverGroupKey = StringUtils.EMPTY;
        }
        return StringUtils.equals(requestGroupKey, serverGroupKey);
    }
}
