package cn.com.duibaboot.ext.autoconfigure.httpclient;

import com.netflix.hystrix.HystrixCommand;
import com.netflix.hystrix.HystrixCommandGroupKey;
import com.netflix.hystrix.HystrixCommandKey;
import com.netflix.hystrix.HystrixCommandProperties;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.HttpHost;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpUriRequest;
import org.springframework.aop.framework.ProxyFactory;

/**
 * 一个HttpClient方法拦截器的实现，用于给HttpClient加上hystrix功能，当某个host下的http调用错误率或超时超过hystrix阈值时，对改host的调用进行熔断
 */
class HttpClientHystrixMethodInterceptor implements HttpClientMethodInterceptor {

	@Override
	public Object invoke(MethodInvocation invocation) throws Throwable {
		if(!"execute".equals(invocation.getMethod().getName())){
			return invocation.proceed();
		}

		Object[] args = invocation.getArguments();
		HttpUriRequest httpUriRequest = null;
		HttpHost httpPost = null;
		for(Object arg : args){
			if(arg instanceof HttpUriRequest){
				httpUriRequest = (HttpUriRequest)arg;
				break;
			}else if(arg instanceof HttpHost){
				httpPost = (HttpHost)arg;
				break;
			}
		}

		String host;
		if(httpUriRequest != null) {
			host = httpUriRequest.getURI().getHost();
		}else if(httpPost != null){
			host = httpPost.getHostName();
		}else{
			return invocation.proceed();
		}

		HttpClientHystrixCommand hystrixCommand = new HttpClientHystrixCommand(invocation, host);

		Object obj = hystrixCommand.execute();
		//由于调用CloseableHttpResponse.close()方法会关闭tcp长连接，而调用 CloseableHttpResponse.getEntity().getContent().close() 只会把连接归还到连接池而不会关闭连接
		//为了提高效率，这里把close方法进行aop，转为调用CloseableHttpResponse.getEntity().getContent().close()
		if(obj instanceof CloseableHttpResponse){
			ProxyFactory proxyFactory = new ProxyFactory();
			proxyFactory.setTarget(obj);
			proxyFactory.addAdvice(new HttpResponseMethodInterceptor());
			return proxyFactory.getProxy();
		}
		return obj;
	}

	@Override
	public int getOrder() {
		return 1;
	}

	private static class HttpResponseMethodInterceptor implements MethodInterceptor{

		@Override
		public Object invoke(MethodInvocation invocation) throws Throwable {
			if("close".equals(invocation.getMethod().getName())
					&& invocation.getMethod().getParameterCount() == 0){
				((CloseableHttpResponse)invocation.getThis()).getEntity().getContent().close();
				return null;
			}
			return invocation.proceed();
		}
	}

	private static class HttpClientHystrixCommand extends HystrixCommand<Object> {

		MethodInvocation invocation;
		String host;

		HttpClientHystrixCommand(MethodInvocation invocation, String host){
			super(HystrixCommand.Setter
					.withGroupKey(HystrixCommandGroupKey.Factory.asKey("httpClient"))
					.andCommandKey(HystrixCommandKey.Factory.asKey("httpClient("+StringUtils.replace(host, ".", ",") +")"))
					.andCommandPropertiesDefaults(HystrixCommandProperties.defaultSetter()
							.withExecutionIsolationStrategy(HystrixCommandProperties.ExecutionIsolationStrategy.SEMAPHORE)
							.withExecutionIsolationSemaphoreMaxConcurrentRequests(200)
							.withExecutionTimeoutEnabled(false)//设置不会超时
					)
			);
			this.invocation = invocation;
			this.host = host;
		}

		@Override
		protected Object run() throws Exception {
			try {
				return invocation.proceed();
			} catch (Exception | Error e){
				throw e;
			} catch (Throwable throwable) {
				throw new RuntimeException(throwable);
			}
		}
	}
}
