package cn.com.duibaboot.ext.autoconfigure.threadpool.proxy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.sleuth.instrument.async.TraceRunnable;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Field;
import java.util.concurrent.BlockingQueue;

/**
 * runnable的包装类，可以监控Runnable从提交到线程池到开始执行花了多久，如果过久则打印warn日志
 * Created by wenqi.huang on 2017/2/5.
 */
public class MonitorRunnable implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(MonitorRunnable.class);

    //一个线程如果从提交线程池到开始执行的时间超过此阈值，则打印warn日志
    static final int MAX_WAIT_TIME_MILLIS = 1000;

    private static final boolean IS_SLEUTH_CLASS_EXISTS;
    private static final Field DELEGATE_FIELD;

    static{
        boolean isSleuthClassExists = false;
        Field delegateField = null;
        try{
            Class.forName("org.springframework.cloud.sleuth.instrument.async.TraceRunnable");
            isSleuthClassExists = true;
            delegateField = ReflectionUtils.findField(TraceRunnable.class, "delegate");
            if(delegateField == null){
                throw new IllegalStateException("field delegate of class `TraceRunnable` not found");
            }
            delegateField.setAccessible(true);
        }catch(Exception e){
        }
        IS_SLEUTH_CLASS_EXISTS = isSleuthClassExists;
        DELEGATE_FIELD = delegateField;
    }

    private Runnable runnable;

    /**
     * 线程池中的queue，当超时的时候可以打印出大小
     */
    private BlockingQueue<Runnable> queue;

    //这个时间用于记录什么时候提交到线程池
    private final long submitTimeMillis;

    public MonitorRunnable(Runnable runnable){
        this.runnable = runnable;
        submitTimeMillis = System.currentTimeMillis();
    }

    public MonitorRunnable(Runnable runnable, BlockingQueue<Runnable> queue){
        this(runnable);
        this.queue = queue;
    }

    public long getSubmitTimeMillis(){
        return submitTimeMillis;
    }

    public Runnable getRunnable() {
        return runnable;
    }

    @Override
    public void run() {
        long waitTimeMillis = System.currentTimeMillis() - submitTimeMillis;
        if(waitTimeMillis >= MAX_WAIT_TIME_MILLIS){
            if(queue == null) {
                logger.error("RunnableClass:{}, ThreadName:{} wait {} ms in queue, maybe corePoolSize is too small.", new Object[]{getRootRunnable().getClass().getName(), Thread.currentThread().getName(), waitTimeMillis});
            }else{
                logger.error("RunnableClass:{}, ThreadName:{} wait {} ms in queue(queueSize:{}), maybe corePoolSize is too small.", new Object[]{getRootRunnable().getClass().getName(), Thread.currentThread().getName(), waitTimeMillis, queue.size()});
            }
        }

        runnable.run();
    }

    /**
     * 获得最底层的runnable对象
     * @return
     */
    protected Runnable getRootRunnable(){
        Runnable rootRunnable = getRunnable();
        if(IS_SLEUTH_CLASS_EXISTS) {
            if (rootRunnable instanceof TraceRunnable) {
                Runnable tmp = (Runnable)ReflectionUtils.getField(DELEGATE_FIELD, rootRunnable);
                if(tmp != null){
                    rootRunnable = tmp;
                }
            }
        }

        return rootRunnable;
    }
}
