package cn.com.duibaboot.ext.autoconfigure.data.etcd.client;

import cn.com.duibaboot.ext.autoconfigure.data.etcd.exception.EtcdOperationException;
import io.etcd.jetcd.ByteSequence;
import io.etcd.jetcd.Client;
import io.etcd.jetcd.KV;
import io.etcd.jetcd.KeyValue;
import io.etcd.jetcd.kv.DeleteResponse;
import io.etcd.jetcd.kv.GetResponse;
import io.etcd.jetcd.kv.PutResponse;
import io.etcd.jetcd.options.DeleteOption;
import io.etcd.jetcd.options.GetOption;
import org.apache.commons.collections.CollectionUtils;

import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

import static io.etcd.jetcd.ByteSequence.from;

/**
 * 封装jetcd的kvClient 提供etcd简单的 put get getWithPrefix delete deleteWithPrefix 操作
 * Created by guoyanfei .
 * 2018/7/12 .
 */
public class JetcdKVClientDelegate implements EtcdKVClientDelegate {

    private static final int      timeout     = 5;
    private static final TimeUnit timeoutUnit = TimeUnit.SECONDS;

    private final KV kvClient;

    public JetcdKVClientDelegate(Client etcdClient) {
        this.kvClient = etcdClient.getKVClient();
    }

    @Override
    public void put(String key, String value) {
        CompletableFuture<PutResponse> putResponseCompletableFuture = this.kvClient.put(from(key, StandardCharsets.UTF_8), from(value, StandardCharsets.UTF_8));
        try {
            putResponseCompletableFuture.get(timeout, timeoutUnit);
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            throw new EtcdOperationException(e);
        }
    }

    @Override
    public String get(String key) {
        List<KeyValue> kvs = this.get(from(key, StandardCharsets.UTF_8), GetOption.DEFAULT);
        if (CollectionUtils.isEmpty(kvs)) {
            return null;
        }
        return kvs.get(0).getValue().toString(StandardCharsets.UTF_8);
    }

    @Override
    public Map<String, String> getWithPrefix(String prefix) {
        GetOption option = GetOption.newBuilder().withPrefix(from(prefix, StandardCharsets.UTF_8)).build();
        List<KeyValue> kvs = this.get(from(prefix, StandardCharsets.UTF_8), option);
        if (CollectionUtils.isEmpty(kvs)) {
            return Collections.emptyMap();
        }
        Map<String, String> resultMap = new HashMap<>();
        for (KeyValue kv : kvs) {
            resultMap.put(kv.getKey().toString(StandardCharsets.UTF_8), kv.getValue().toString(StandardCharsets.UTF_8));
        }
        return resultMap;
    }

    private List<KeyValue> get(ByteSequence key, GetOption option) {
        CompletableFuture<GetResponse> getResponseCompletableFuture = this.kvClient.get(key, option);
        GetResponse getResponse;
        try {
            getResponse = getResponseCompletableFuture.get(timeout, timeoutUnit);
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            throw new EtcdOperationException(e);
        }
        if (getResponse == null) {
            return Collections.emptyList();
        }
        return getResponse.getKvs();
    }

    @Override
    public void delete(String key) {
        this.delete(from(key, StandardCharsets.UTF_8), DeleteOption.DEFAULT);
    }

    @Override
    public void deleteWithPrefix(String prefix) {
        DeleteOption option = DeleteOption.newBuilder().withPrefix(from(prefix, StandardCharsets.UTF_8)).build();
        this.delete(from(prefix, StandardCharsets.UTF_8), option);
    }

    private void delete(ByteSequence key, DeleteOption option) {
        CompletableFuture<DeleteResponse> deleteResponseCompletableFuture = this.kvClient.delete(key, option);
        try {
            deleteResponseCompletableFuture.get(timeout, timeoutUnit);
        } catch (InterruptedException | ExecutionException | TimeoutException e) {
            throw new EtcdOperationException(e);
        }
    }

    public static void main(String[] args) {
        JetcdKVClientDelegate delegate = new JetcdKVClientDelegate(Client.builder().endpoints("http://172.16.80.120:2379").build());//NOSONAR
        System.out.println(delegate.getWithPrefix("/config"));
        System.out.println(delegate.get("/config/zuul-server/zuul/routes/millionaire-web222/strip-prefix"));
    }
}
