package cn.com.duibaboot.ext.autoconfigure.dns;

import java.lang.reflect.Field;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CompletionException;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import javax.annotation.PostConstruct;

import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import sun.net.spi.nameservice.NameService;

import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Configuration;

/**
 * 自动配置DNS缓存（支持异步刷新）
 * @author hwq
 */
@Configuration
@ConditionalOnProperty(value="duiba.dns.cache.enabled", havingValue = "true", matchIfMissing = false)
@Slf4j
public class DnsCacheAutoConfiguration {

	@PostConstruct
	public void init() {
		try {
			Class.forName("java.net.InetAddress");
		}
		catch (ClassNotFoundException e) {
			//ignore
		}

		try {
			Field field = InetAddress.class.getDeclaredField("nameServices");
			field.setAccessible(true);
			List<NameService> nameServices = (List<NameService>)field.get(null);
			List<NameService> nameServicesNew = new ArrayList<>();
			for(NameService nameService : nameServices) {
				nameServicesNew.add(new CachedNameServiceWrapper(nameService));
			}
			int i = 0;
			for(NameService nameService : nameServicesNew) {
				nameServices.set(i++, nameService);
			}
		}
		catch (NoSuchFieldException | IllegalAccessException e) {
			log.warn("替换DNS解析器失败！，貌似JDK新版本改变了？", e);
		}
	}

	/**
	 * DNS缓存，内部把DNS永久缓存起来，内部走异步refresh,保证对于高频访问的域名，一直有缓存存在，以避免偶尔的unknown host异常，并加快dns解析速度;
	 * JDK内部默认已经有30秒的缓存，所以对于每个域名，CachedNameServiceWrapper.lookupAllHostAddr30秒内最多会被调用一次。
	 */
	private static final class CachedNameServiceWrapper implements NameService {

		private static final ThreadPoolExecutor executorService;
		static{
			executorService = new ThreadPoolExecutor(1, 20,
					60, TimeUnit.SECONDS,
					new SynchronousQueue<>(),
					new NamedThreadFactory("DnsResolve", true), new ThreadPoolExecutor.CallerRunsPolicy());
			executorService.allowCoreThreadTimeOut(true);
		}

		private LoadingCache<String, InetAddress[]> dnsCache;

		private NameService delegate;

		public CachedNameServiceWrapper(NameService delegate) {
			this.delegate = delegate;
			dnsCache = Caffeine.newBuilder()
					.expireAfterAccess(60, TimeUnit.MINUTES)
					.refreshAfterWrite(10, TimeUnit.SECONDS)
					.maximumSize(10000)
					.initialCapacity(1000)
					.executor(executorService)
					.build(key -> delegate.lookupAllHostAddr(key));
		}

		@Override
		public InetAddress[] lookupAllHostAddr(String host) throws UnknownHostException {
			try {
				return dnsCache.get(host);
			}catch(CompletionException e){
				if(e.getCause() != null && e.getCause() instanceof UnknownHostException) {
					throw (UnknownHostException) e.getCause();//UnknownHostException
				} else if(e.getCause() != null && e.getCause() instanceof RuntimeException) {
					throw (RuntimeException)e.getCause();
				} else {
					throw e;
				}
			}
		}

		@Override
		public String getHostByAddr(byte[] bytes) throws UnknownHostException {
			return delegate.getHostByAddr(bytes);
		}
	}
}
