package cn.com.duibaboot.ext.autoconfigure.elasticjob;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import com.netflix.discovery.EurekaClient;
import io.elasticjob.lite.api.JobScheduler;
import io.elasticjob.lite.config.LiteJobConfiguration;
import io.elasticjob.lite.internal.config.LiteJobConfigurationGsonFactory;
import io.elasticjob.lite.internal.schedule.JobRegistry;
import io.elasticjob.lite.internal.storage.JobNodePath;
import io.elasticjob.lite.reg.base.CoordinatorRegistryCenter;
import io.elasticjob.lite.reg.zookeeper.ZookeeperConfiguration;
import io.elasticjob.lite.reg.zookeeper.ZookeeperRegistryCenter;
import lombok.extern.slf4j.Slf4j;
import org.apache.http.client.HttpClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.cloud.client.loadbalancer.RestTemplateCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;
import org.springframework.http.client.HttpComponentsClientHttpRequestFactory;
import org.springframework.http.converter.ByteArrayHttpMessageConverter;
import org.springframework.util.ClassUtils;
import org.springframework.util.ReflectionUtils;
import org.springframework.web.client.RestTemplate;

import java.lang.reflect.Field;
import java.util.List;
import java.util.Map;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2019-08-22 20:23
 * @descript:
 * @version: 1.0
 */
@Slf4j
@Configuration
@ConditionalOnClass({JobScheduler.class, HttpClient.class, EurekaClient.class})
public class ElasticjobMonitorAutoConfiguration {

    @Autowired
    private ElasticjobMonitorManager elasticjobMonitorManager;

    @EventListener(MainContextRefreshedEvent.class)
    public void init() {

        JobRegistry jobRegistry = JobRegistry.getInstance();
        if (jobRegistry == null) {
            return;
        }

        try {

            Field regCenterMapField = ReflectionUtils.findField(JobRegistry.class, "regCenterMap");
            ReflectionUtils.makeAccessible(regCenterMapField);
            Map<String, CoordinatorRegistryCenter> centerMap = (Map<String, CoordinatorRegistryCenter>) ReflectionUtils.getField(regCenterMapField, jobRegistry);
            if (centerMap == null || centerMap.size() == 0) {
                return;
            }

            boolean isSetJobName = false;
            for (Map.Entry<String, CoordinatorRegistryCenter> entry : centerMap.entrySet()) {

                ZookeeperRegistryCenter center = (ZookeeperRegistryCenter) entry.getValue();

                if (!isSetJobName) {

                    Field zkConfigField = ReflectionUtils.findField(ZookeeperRegistryCenter.class, "zkConfig");
                    ReflectionUtils.makeAccessible(zkConfigField);
                    ZookeeperConfiguration configuration = (ZookeeperConfiguration) ReflectionUtils.getField(zkConfigField, center);
                    elasticjobMonitorManager.setJobNamespace(configuration.getNamespace());
                    isSetJobName = true;
                }

                String jobName = entry.getKey();
                JobNodePath jobNodePath = new JobNodePath(jobName);
                String json = center.getDirectly(jobNodePath.getConfigNodePath());
                LiteJobConfiguration liteJobConfiguration = LiteJobConfigurationGsonFactory.fromJson(json);
                elasticjobMonitorManager.putJobCronMap(jobName, liteJobConfiguration.getTypeConfig().getCoreConfig().getCron());
            }


        } catch (Exception e) {
            log.info("初始化作业的信息异常", e);
        }

        boolean isPresent = ClassUtils.isPresent("io.elasticjob.autoconfigure.eventbus.JobEventCacheStorage", null);
        if(!isPresent) {
            log.warn("未发现JobEventCacheStorage实例，请去除elastic-job-spring-boot-starter的版本号指定，由ext来管理版本，否则无法监控job的历史轨迹和异常报警");
        }
    }

    @Configuration
    public static class ElasticjobMonitorManagerConfiguration {

        @Bean
        public ElasticjobMonitorMvcEndpoint elasticjobMonitorEndpointConfiguration() {
            return new ElasticjobMonitorMvcEndpoint();
        }

        @Bean
        public ElasticjobMonitorManager elasticjobMonitorManagerConfiguration() {
            return new ElasticjobMonitorManager();
        }

        @Bean
        public ElasticjobRecordCompent elasticjobRecordCompent(HttpClient httpClient,
                                                               ElasticjobMonitorManager monitorManager,
                                                               List<RestTemplateCustomizer> customizers,
                                                               @Value("${spring.application.name}") String currentAppName) {
            RestTemplate restTemplate = elasticjobPushRestTemplate(httpClient);
            for (RestTemplateCustomizer customizer : customizers) {
                customizer.customize(restTemplate);
            }
            return new ElasticjobRecordCompent(restTemplate, monitorManager, currentAppName);
        }

        private RestTemplate elasticjobPushRestTemplate(HttpClient httpClient) {
            HttpComponentsClientHttpRequestFactory httpRequestFactory = new HttpComponentsClientHttpRequestFactory();
            httpRequestFactory.setConnectionRequestTimeout(3000);
            httpRequestFactory.setConnectTimeout(3000);
            httpRequestFactory.setReadTimeout(60000);
            httpRequestFactory.setHttpClient(httpClient);
            RestTemplate template = new RestTemplate(httpRequestFactory);
            template.getMessageConverters().add(new ByteArrayHttpMessageConverter());
            return template;
        }
    }
}
