package cn.com.duibaboot.ext.autoconfigure.elasticjob;

import cn.com.duiba.boot.utils.NetUtils;
import com.google.common.base.Joiner;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import io.elasticjob.autoconfigure.eventbus.JobEventCacheStorage;
import io.elasticjob.lite.api.ElasticJob;
import io.elasticjob.lite.api.ShardingContext;
import io.elasticjob.lite.api.dataflow.DataflowJob;
import io.elasticjob.lite.api.script.ScriptJob;
import io.elasticjob.lite.api.simple.SimpleJob;
import io.elasticjob.lite.event.type.JobExecutionEvent;
import io.elasticjob.lite.executor.ShardingContexts;
import io.elasticjob.lite.util.concurrent.ExecutorServiceObject;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.core.annotation.AnnotationUtils;

import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2019-08-22 20:27
 * @descript:
 * @version: 1.0
 */
@Slf4j
public class ElasticjobMonitorManager {

    @Autowired
    private ApplicationContext applicationContext;

    private Map<String, String> jobCronMap = Maps.newConcurrentMap();

    private final Set<ElasticJob> runingJob = Sets.newConcurrentHashSet();

    private volatile String jobNamespace;

    private final ExecutorService executorService = new ExecutorServiceObject("monitor-job-trigger", Runtime.getRuntime().availableProcessors() * 2).createExecutorService();

    void setJobNamespace(String namespace) {
        jobNamespace = namespace;
    }

    String getJobNamespace() {
        return jobNamespace;
    }

    void putJobCronMap(String jobName, String cron) {
        jobCronMap.put(jobName, cron);
    }

    String getJobCron(String jobName) {
        return jobCronMap.get(jobName);
    }

    String trigger(String jobClassName, List<Integer> shardingItems) {

        if (CollectionUtils.isEmpty(shardingItems)) {
            return "没有分片信息";
        }

        ElasticJob elasticJob = null;
        String message = "触发作业异常";
        try {
            Class<?> jobClass = Class.forName(jobClassName);
            io.elasticjob.autoconfigure.annotation.ElasticJob an = AnnotationUtils.findAnnotation(jobClass, io.elasticjob.autoconfigure.annotation.ElasticJob.class);
            if (an == null || StringUtils.isEmpty(an.name())) {
                return "目前只支持有job注解的触发";
            }

            Map<String, ?> beanMap = applicationContext.getBeansOfType(jobClass);
            if (beanMap == null || beanMap.size() == 0) {
                return "没有找到可以触发的job示例";
            }

            Map.Entry<String, ?> entry = beanMap.entrySet().iterator().next();
            if (!(entry.getValue() instanceof ElasticJob)) {
                return "没有找到可以触发的job示例";
            }

            elasticJob = (ElasticJob) entry.getValue();
            if (runingJob.contains(elasticJob)) {
                return "当前job正在运行，请稍后再试";
            }

            if (elasticJob instanceof ScriptJob) {
                return "暂时不支持手动调度ScriptJob";
            }

            runingJob.add(elasticJob);
            if (elasticJob instanceof SimpleJob) {
                simpleProcess(elasticJob, shardingItems, an.name());
            } else if (elasticJob instanceof DataflowJob) {
                dataProcess(elasticJob, shardingItems, an.name());
            }

            message = "触发作业成功";

        } catch (Exception e) {
            log.error("触发作业异常", e);
        } finally {

            if (elasticJob != null) {
                runingJob.remove(elasticJob);
            }
        }

        return message;
    }

    private void simpleProcess(ElasticJob elasticJob, List<Integer> shardingItems, String jobName) {

        String taskId = this.buildTaskId(jobName, shardingItems);
        ShardingContexts empty = new ShardingContexts(taskId, jobName, 0, null, Maps.newHashMap());
        final CountDownLatch latch = new CountDownLatch(shardingItems.size());
        shardingItems.forEach(shardingItem -> executorService.submit(() -> {

            try {
                JobExecutionEvent jobExecutionEvent = new JobExecutionEvent(taskId, jobName, null, shardingItem);
                ((SimpleJob) elasticJob).execute(new ShardingContext(empty, shardingItem));
                jobExecutionEvent = jobExecutionEvent.executionSuccess();
                JobEventCacheStorage.addJobExecutionEvent(jobExecutionEvent);
            } finally {
                latch.countDown();
            }
        }));

        try {
            latch.await();
        } catch (final InterruptedException ex) {
            Thread.currentThread().interrupt();
        }
    }

    private String buildTaskId(final String jobName, final List<Integer> shardingItems) {
        String instanceId = String.format("%s@-@1", NetUtils.getLocalIp());
        return Joiner.on("@-@").join(jobName, Joiner.on(",").join(shardingItems), "READY", instanceId);
    }

    private void dataProcess(ElasticJob elasticJob, List<Integer> shardingItems, String jobName) {

        String taskId = this.buildTaskId(jobName, shardingItems);
        final CountDownLatch latch = new CountDownLatch(shardingItems.size());
        final ShardingContexts empty = new ShardingContexts(null, null, 0, null, Maps.newHashMap());
        shardingItems.forEach(shardingItem -> executorService.submit(() -> {

            try {
                JobExecutionEvent jobExecutionEvent = new JobExecutionEvent(taskId, jobName, null, shardingItem);
                DataflowJob job = (DataflowJob) elasticJob;
                ShardingContext context = new ShardingContext(empty, shardingItem);
                List<?> data = job.fetchData(context);
                job.processData(context, data);
                jobExecutionEvent = jobExecutionEvent.executionSuccess();
                JobEventCacheStorage.addJobExecutionEvent(jobExecutionEvent);
            } finally {
                latch.countDown();
            }
        }));

        try {
            latch.await();
        } catch (final InterruptedException ex) {
            Thread.currentThread().interrupt();
        }

    }
}
