package cn.com.duibaboot.ext.autoconfigure.elasticjob;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import cn.com.duiba.wolf.utils.BeanUtils;
import cn.com.duibaboot.ext.autoconfigure.core.EarlyClose;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import io.elasticjob.autoconfigure.eventbus.JobEventCacheStorage;
import io.elasticjob.lite.event.type.JobExecutionEvent;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.quartz.CronExpression;
import org.springframework.context.event.EventListener;
import org.springframework.http.*;
import org.springframework.util.ClassUtils;
import org.springframework.util.CollectionUtils;
import org.springframework.web.client.RestTemplate;

import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;

/**
 * @author: <a href="http://www.panaihua.com">panaihua</a>
 * @date: 2020-12-17 14:35
 * @descript:
 * @version: 1.0
 */

@Slf4j
public class ElasticjobRecordCompent extends EarlyClose {

    private final RestTemplate restTemplate;

    private final String currentAppName;

    private final ElasticjobMonitorManager monitorManager;

    private final HttpHeaders jsonHeader;

    private final String serverUrl;

    private final ScheduledExecutorService pushScheduled = Executors.newSingleThreadScheduledExecutor(new NamedThreadFactory("duiba-elasticjob-push", true));

    public ElasticjobRecordCompent(RestTemplate restTemplate, ElasticjobMonitorManager monitorManager,
                                   String currentAppName) {
        this.restTemplate = restTemplate;
        this.monitorManager = monitorManager;
        this.currentAppName = currentAppName;
        jsonHeader = new HttpHeaders();
        jsonHeader.setContentType(MediaType.APPLICATION_JSON);
        this.serverUrl = String.format("http://prism/prism/elasticjob/push/consume/%s", NetUtils.getLocalIp());
    }

    @EventListener(MainContextRefreshedEvent.class)
    public void startPush() {

        if (pushScheduled.isShutdown()) {
            return;
        }

        pushScheduled.scheduleAtFixedRate(() -> {

            try {
                this.pushTrackEvents();
            } catch (Exception e) {
                log.info("推送elasticjob数据异常", e);
            }

        }, 1, 5, TimeUnit.MINUTES);

    }

    public void pushTrackEvents() {

        try {
            List<JobExecutionTrackEvent> events = this.builTrackEvents();
            if (CollectionUtils.isEmpty(events)) {
                return;
            }

            HttpEntity<String> request = new HttpEntity<>(JSONObject.toJSONString(events), jsonHeader);

            ResponseEntity<String> resp = restTemplate.postForEntity(serverUrl, request, String.class);
            if (resp.getStatusCode() != HttpStatus.OK) {
                log.warn("推送elasticjob数据异常，返回错误码：{}", resp.getStatusCode().value());
            }

        } catch (Exception e) {
            log.warn("推送elasticjob数据异常", e);
        }
    }

    private List<JobExecutionTrackEvent> builTrackEvents() {

        boolean isPresent = ClassUtils.isPresent("io.elasticjob.autoconfigure.eventbus.JobEventCacheStorage", null);
        if (!isPresent) {
            return Collections.emptyList();
        }

        List<JobExecutionEvent> events = JobEventCacheStorage.consumeEvent();
        List<JobExecutionTrackEvent> trackEvents = Lists.newArrayListWithCapacity(events.size());
        events.forEach(event -> {

            JobExecutionTrackEvent trackEvent = BeanUtils.copy(event, JobExecutionTrackEvent.class);
            trackEvent.setFailureCause(event.getFailureCause());
            trackEvent.setAppName(currentAppName);
            String cron = monitorManager.getJobCron(event.getJobName());

            if (StringUtils.isBlank(cron)) {
                return;
            }

            try {

                CronExpression cronSequenceGenerator = new CronExpression(cron);
                Date nextTime = cronSequenceGenerator.getNextValidTimeAfter(trackEvent.getStartTime());
                trackEvent.setNextTime(nextTime);
            } catch (Exception e) {
                log.warn("解析任务的下次执行时间异常, jobName:{}", event.getJobName(), e);
            }

            trackEvents.add(trackEvent);

        });

        return trackEvents;
    }

    private void shutdown() {
        log.info("begin to shutdown elasticjob push server...");
        pushScheduled.shutdown();
        try {

            if (!pushScheduled.awaitTermination(5, TimeUnit.SECONDS)) {
                log.info("等待超过5秒任务强制关闭");
                pushScheduled.shutdownNow();

                if (!pushScheduled.awaitTermination(3, TimeUnit.SECONDS)) {
                    log.info("当前任务无法强制关闭");
                }

            }
        } catch (InterruptedException e) {
            pushScheduled.shutdownNow();
            Thread.currentThread().interrupt();
        }

        pushTrackEvents();
    }

    @Override
    public void stop() {
        this.shutdown();
    }
}
