package cn.com.duibaboot.ext.autoconfigure.security;

import cn.com.duibaboot.ext.autoconfigure.core.utils.SpringBootUtils;
import net.bytebuddy.dynamic.ClassFileLocator;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationListener;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.ContextRefreshedEvent;
import org.springframework.core.Ordered;
import javax.annotation.PostConstruct;
import javax.security.auth.AuthPermission;
import java.io.FilePermission;
import java.io.SerializablePermission;
import java.lang.reflect.ReflectPermission;
import java.net.NetPermission;
import java.net.SocketPermission;
import java.security.AccessControlException;
import java.security.AllPermission;
import java.security.Permission;
import java.security.SecurityPermission;
import java.sql.SQLPermission;
import java.util.Arrays;
import java.util.HashSet;
import java.util.PropertyPermission;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.logging.LoggingPermission;

/**
 * 自动配置SecurityManager，防止shell攻击（即使有fastjson等存在shell漏洞的框架也能有效防止）
 * http://www.cnblogs.com/yiwangzhibujian/p/6207212.html
 * Created by wenqi.huang on 2017/4/5.
 */
@Configuration
@ConditionalOnProperty(name = "duiba.securitymanager.enable", havingValue="true",  matchIfMissing = true)
@EnableConfigurationProperties(SecurityProperties.class)
public class SecurityManagerAutoConfiguration {

    private static final Logger logger = LoggerFactory.getLogger(SecurityManagerAutoConfiguration.class);

    public static final int ORDER_OF_CONTEXT_REFRESHED_EVENT = -10;

    static Set<String> forbiddenPermNames = new HashSet<>(10);

    @Autowired
    private SecurityProperties securityProperties;

    private Set<String> whiteShells = new HashSet<>();

    @PostConstruct
    public void init(){
        whiteShells.addAll(Arrays.asList(new String[]{
                "/usr/bin/id", "/bin/id", "id", "/usr/xpg4/bin/id", //netty需要执行的命令
                "setsid", //hbase需要执行的命令
                "jstat", "jmap"
        }));

        String[] forbiddenShells = new String[]{"sh","bash","source","exec","fork"};//必须禁止的shell
        String str = securityProperties.getWhiteShellList();
        String[] arr = StringUtils.split(str, ',');
        if(arr == null){
            return;
        }
        for(String shell : arr){
            if(isForbiddenShell(shell, forbiddenShells)){
                logger.warn("你设置的shell白名单(duiba.securitymanager.whiteShellList)中包含过于危险的shell：`{}`, 已排除.", shell);
            } else{
                whiteShells.add(shell);
            }
        }
    }

    private boolean isForbiddenShell(String shell, String[] forbiddenShells){
        for(String forbiddenShell : forbiddenShells) {
            if (forbiddenShell.equalsIgnoreCase(shell)){
                return true;
            }
        }
        return false;
    }

    static{
        //forbiddenPermNames.add("stopThread");
        forbiddenPermNames.add("queuePrintJob");
        //访问https url时jdk会创建 JCESecurityManager
        //forbiddenPermNames.add("createSecurityManager");

        //测试模式不屏蔽下面这些权限。否则运行单测会报错
        if(!SpringBootUtils.isUnitTestMode()) {//,"org.gradle.api.internal.tasks.testing.worker.TestWorker"
            forbiddenPermNames.add("setIO");
            forbiddenPermNames.add("setSecurityManager");
        }
    }

    //指定的类都不存在时返回true，有一个存在则返回false
    private static boolean conditionalOnMissingClass(String... classNames) {
        boolean existsOne = false;
        for(String className : classNames) {
            try {
                Class.forName(className);
                existsOne = true;
                break;
            } catch (ClassNotFoundException e) {
                //Ignore
            }
        }

        return !existsOne;
    }

    @Bean
    public DuibaSecurityManagerConfiguarApplicationListener duibaSecurityManagerConfiguarApplicationListener(){
        return new DuibaSecurityManagerConfiguarApplicationListener();
    }


    /**
     * A circularity lock is responsible for preventing that a {@link ClassFileLocator} is used recursively.
     * This can happen when a class file transformation causes another class to be loaded. Without avoiding
     * such circularities, a class loading is aborted by a {@link ClassCircularityError} which causes the
     * class loading to fail.
     */
    interface CircularityLock {

        /**
         * Attempts to acquire a circularity lock.
         *
         * @return {@code true} if the lock was acquired successfully, {@code false} if it is already hold.
         */
        boolean acquire();

        /**
         * Releases the circularity lock if it is currently acquired.
         */
        void release();

        /**
         * An inactive circularity lock which is always acquirable.
         */
        enum Inactive implements CircularityLock {

            /**
             * The singleton instance.
             */
            INSTANCE;

            @Override
            public boolean acquire() {
                return true;
            }

            @Override
            public void release() {
                /* do nothing */
            }
        }

        /**
         * A default implementation of a circularity lock. Since class loading already synchronizes on a class loader,
         * it suffices to apply a thread-local lock.
         */
        class Default extends ThreadLocal<Boolean> implements CircularityLock {

            /**
             * Indicates that the circularity lock is not currently acquired.
             */
            private static final Boolean NOT_ACQUIRED = null;

            @Override
            public boolean acquire() {
                if (get() == NOT_ACQUIRED) {
                    set(true);
                    return true;
                } else {
                    return false;
                }
            }

            @Override
            public void release() {
                set(NOT_ACQUIRED);
            }
        }

        /**
         * A circularity lock that holds a global monitor and does not permit concurrent access.
         */
        class Global implements CircularityLock {

            /**
             * The lock to hold.
             */
            private final Lock lock;

            /**
             * The time to wait for the lock.
             */
            private final long time;

            /**
             * The time's time unit.
             */
            private final TimeUnit timeUnit;

            /**
             * Creates a new global circularity lock that does not wait for a release.
             */
            public Global() {
                this(0, TimeUnit.MILLISECONDS);
            }

            /**
             * Creates a new global circularity lock.
             *
             * @param time     The time to wait for the lock.
             * @param timeUnit The time's time unit.
             */
            public Global(long time, TimeUnit timeUnit) {
                lock = new ReentrantLock();
                this.time = time;
                this.timeUnit = timeUnit;
            }

            @Override
            public boolean acquire() {
                try {
                    return time == 0
                            ? lock.tryLock()
                            : lock.tryLock(time, timeUnit);
                } catch (InterruptedException ignored) {
                    return false;
                }
            }

            @Override
            public void release() {
                lock.unlock();
            }
        }
    }


    class DuibaSecurityManagerConfiguarApplicationListener implements ApplicationListener<ContextRefreshedEvent>, Ordered {

        private boolean flag = true;

        @Override
        public void onApplicationEvent(ContextRefreshedEvent applicationStartedEvent) {
			//windows系统不设置SecurityManager,以防止java.lang.ClassCircularityError
			String os = StringUtils.defaultString(System.getProperty("os.name"));
			if(os.toLowerCase().contains("windows")){
				return;
			}

            if(!flag) {
                return;
            }
            if(System.getSecurityManager() == null) {
                System.setSecurityManager(new CustomSecurityManager());
            }

            flag = false;
        }

        @Override
        public int getOrder() {
            return ORDER_OF_CONTEXT_REFRESHED_EVENT;
        }
    }

    class CustomSecurityManager extends BaseSecurityManager {

		private CircularityLock lock = new CircularityLock.Default();

        private boolean isInWhiteShellList(String cmd){
            if(whiteShells.contains(cmd)){
                return true;
            }
            return false;
        }

        @Override
        public void checkExec(String cmd) {
            if(isInWhiteShellList(cmd)){
                return;
            }

            //不允许执行shell脚本
            try {
                super.checkExec(cmd);
            }catch(AccessControlException e){
                logger.warn("some one try to execute shell:`{}`, block it", cmd);
                throw e;
            }
        }

        @Override
        public void checkPermission(Permission perm) {
            if (lock.acquire()){ //防止java.lang.ClassCircularityError
                try {
                    checkPermissionInner(perm);
                }finally{
                    lock.release();
                }
            }

            //如果没有获得锁，则不检查权限
            return;
        }

        private void checkPermissionInner(Permission perm) {
//                                Class<?>[] classes = this.getClassContext();
//                                StackTraceElement[] st = new RuntimeException().getStackTrace();
            //这里判断Permission类型的方法，如果换成Set.contains判断，速度会降低2000倍，所以用if来判断

            if(perm.getName().equals("getClassLoader")
                    || perm.getName().equals("getProtectionDomain")
                    || perm.getName().equals("accessDeclaredMembers")
                    || perm.getName().equals("<all permissions>")
                    || perm.getClass().getName().equals("java.lang.RuntimePermission")){//NOSONAR
                return;
            }

            //这里不能使用instanceof来判断，如果这么做了，会与bytebuddy冲突，导致抛出java.lang.ClassCircularityError
            if(perm.getClass().getName().equals("java.io.FilePermission")){//NOSONAR
                FilePermission fp = (FilePermission)perm;
                if(fp.getActions().equals("execute")){
                    super.checkPermission(perm);
                    return;
                }
            }

            if(perm instanceof AllPermission
                    || perm instanceof PropertyPermission
                    || perm instanceof SerializablePermission
                    || perm instanceof ReflectPermission
                    || perm instanceof NetPermission
                    || perm instanceof SQLPermission
                    || perm instanceof SecurityPermission
                    || perm instanceof LoggingPermission
                    || perm instanceof AuthPermission
                    || perm instanceof SocketPermission){
                return;
            }

            if(forbiddenPermNames.contains(perm.getName())) {
                super.checkPermission(perm);
                return;
            }
        }

        @Override
        public void checkConnect(String host, int port) {
            //允许连接任意host
        }

        @Override
        public void checkConnect(String host, int port, Object context) {
            //允许连接任意host
        }

    }
}
