package cn.com.duibaboot.ext.autoconfigure.dubbo.init;

import cn.com.duiba.wolf.threadpool.NamedThreadFactory;
import cn.com.duibaboot.ext.autoconfigure.dubbo.core.DubboSpecifiedBeanPostProcessor;
import org.apache.dubbo.config.spring.context.event.DubboBootstrapStatedEvent;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationListener;
import org.springframework.core.Ordered;

import java.util.Map;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;

/**
 * Created by wenqi.huang on 2017/6/16.
 */
public class DubboBootstrapListenerForAsyncSpecified implements ApplicationListener<DubboBootstrapStatedEvent>, Ordered {

    private static final Logger logger = LoggerFactory.getLogger(DubboBootstrapListenerForAsyncSpecified.class);


    private Map<DubboSpecifiedBeanPostProcessor, Map<String, Object>> dubboAsyncProcessor2BeansMap;

    public DubboBootstrapListenerForAsyncSpecified(Map<DubboSpecifiedBeanPostProcessor, Map<String, Object>> dubboAsyncProcessor2BeansMap) {
        this.dubboAsyncProcessor2BeansMap = dubboAsyncProcessor2BeansMap;
    }

    @Override
    public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE;
    }

    @Override
    public void onApplicationEvent(DubboBootstrapStatedEvent event) {

        //开启线程池用于异步方法，执行完成后立即销毁
        ExecutorService executorService = Executors.newFixedThreadPool(5, new NamedThreadFactory("DuibaBootDubboInitServerThread-"));

        //异步化执行所有DubboSpecifiedBeanPostProcessor
        int totalBean2ProcessCount = 0;
        for (Map<String, Object> beanMap : dubboAsyncProcessor2BeansMap.values()) {
            totalBean2ProcessCount += beanMap.size();
        }

        final CountDownLatch countDownLatch = new CountDownLatch(totalBean2ProcessCount);

        for (Map.Entry<DubboSpecifiedBeanPostProcessor, Map<String, Object>> entry : dubboAsyncProcessor2BeansMap.entrySet()) {
            Map<String, Object> beanMap = entry.getValue();
            final DubboSpecifiedBeanPostProcessor processor = entry.getKey();
            for (final Map.Entry<String, Object> beanEntry : beanMap.entrySet()) {
                executorService.submit(() -> {
                    try {
                        processor.postProcessAfterInitialization(beanEntry.getValue(), beanEntry.getKey());
                    } catch (Exception e) {
                        logger.warn("Dubbo Init failed", e);
                    } finally {
                        countDownLatch.countDown();
                    }
                });
            }
        }

        //阻塞住spring流程，确保异步初始化全部执行完成再继续(最多等三秒，以防有线程执行时间太长阻塞启动)
        try {
            boolean success = countDownLatch.await(3, TimeUnit.SECONDS);
            if (!success) {
                logger.info("部分功能没有初始化成功");
            }
        } catch (InterruptedException e) {
            //Ignore
            Thread.currentThread().interrupt();
        }

        executorService.shutdown();
    }

}
