package cn.com.duibaboot.ext.autoconfigure.dubbo.sleuth;

import brave.ErrorParser;
import brave.Span;
import brave.Tracer;
import brave.Tracing;
import brave.propagation.TraceContext;
import brave.propagation.TraceContextOrSamplingFlags;
import cn.com.duiba.boot.perftest.PerfTestContext;
import cn.com.duiba.boot.utils.MainApplicationContextHolder;
import org.apache.dubbo.common.URL;
import org.apache.dubbo.common.constants.CommonConstants;
import org.apache.dubbo.common.extension.Activate;
import org.apache.dubbo.rpc.*;
import org.apache.dubbo.rpc.protocol.dubbo.FutureAdapter;
import org.apache.dubbo.rpc.support.RpcUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.StringUtils;

import java.util.concurrent.Future;

/**
 * sleuth对dubbo的支持
 * 参考 https://github.com/openzipkin/brave/blob/master/instrumentation/dubbo/src/main/java/brave/dubbo/TracingFilter.java
 *
 * @author wenqi.huang
 */
@Activate(group = {CommonConstants.PROVIDER, CommonConstants.CONSUMER},order = 1)
public class SleuthDubboFilter implements Filter {

    private static final Logger log = LoggerFactory.getLogger(SleuthDubboFilter.class);
    private static final boolean HAS_TRACER_CLASS;

    private static volatile Tracer tracer;
    private static volatile Tracing tracing;
    private static volatile ErrorParser errorParser;

    private static volatile TraceContext.Injector<DubboRequestTextMap> injector;
    private static volatile TraceContext.Extractor<DubboRequestTextMap> extractor;

    static {
        boolean hasTracerClass = true;
        try {
            Class.forName("org.springframework.cloud.sleuth.Tracer");
        }
        catch (ClassNotFoundException e) {
            hasTracerClass = false;
        }
        HAS_TRACER_CLASS = hasTracerClass;
    }

    private static Tracer getTracer(){
        if (tracer == null) {
            if(MainApplicationContextHolder.getApplicationContext() != null) {
                tracer = MainApplicationContextHolder.getApplicationContext().getBean(Tracer.class);
            }
        }
        return tracer;
    }

    private static Tracing getTracing(){
        if (tracing == null) {
            if(MainApplicationContextHolder.getApplicationContext() != null) {
                tracing = MainApplicationContextHolder.getApplicationContext().getBean(Tracing.class);
            }
        }
        return tracing;
    }

    private static ErrorParser getErrorParser(){
        if (errorParser == null) {
            if(MainApplicationContextHolder.getApplicationContext() != null) {
                errorParser = MainApplicationContextHolder.getApplicationContext().getBean(ErrorParser.class);
            }
        }
        return errorParser;
    }

    private static TraceContext.Injector<DubboRequestTextMap> getInjector(){
        if (injector == null) {
            if(getTracing() != null) {
                injector = getTracing().propagation().injector((carrier, key, value) -> {
                    carrier.put(key, value);
                });
            }
        }
        return injector;
    }

    private static TraceContext.Extractor<DubboRequestTextMap> getExtractor(){
        if (extractor == null) {
            if(getTracing() != null) {
                extractor = getTracing().propagation().extractor((carrier, key) -> carrier.get(key));
            }
        }
        return extractor;
    }

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException { //NOSONAR
        if (!HAS_TRACER_CLASS) {
            return invoker.invoke(invocation);
        }
        if (getTracer() == null) {
            return invoker.invoke(invocation);
        }

        RpcContext rpcContext = RpcContext.getContext();
        URL url = invoker.getUrl();
        String sideKey = url.getParameter(CommonConstants.SIDE_KEY);
        String spanName = "dubbo:/" + invoker.getInterface().getSimpleName()+"/"+invocation.getMethodName();
        if (CommonConstants.CONSUMER_SIDE.equals(sideKey)) {
            Span span = getTracer().nextSpan().name(spanName)
                    .kind(Span.Kind.CLIENT)
                    .start();
            try(Tracer.SpanInScope scope = tracer.withSpanInScope(span)) {
                getInjector().inject(span.context(), new DubboRequestTextMap(RpcContext.getContext()));
                if(!span.isNoop()) {
                    span.tag("lc", "dubbo");//本地组件名
//                span.tag(Span.SPAN_PEER_SERVICE_TAG_NAME, "dubbo");//远程服务名
                    span.tag("peer.host", RpcContext.getContext().getRemoteHost() + ":" + RpcContext.getContext().getRemotePort());//远程host
                    span.tag("dubbo.invokeMethod", invoker.getInterface().getSimpleName() + "." + invocation.getMethodName());//method
                    span.tag("client.isPerfTest", Boolean.toString(PerfTestContext.isCurrentInPerfTestMode()));
                    span.tag("client.thread", Thread.currentThread().getName());
                }
                Result result = invoker.invoke(invocation);
                if(result.hasException()) {
                    getErrorParser().error(result.getException(), span);
                }
                return result;
            } catch(Exception e) {
                getErrorParser().error(e, span);
                throw e;
            } finally {
                span.finish();
            }
        } else { //PROVIDER_SIDE
            TraceContextOrSamplingFlags extracted = getExtractor().extract(new DubboRequestTextMap(RpcContext.getContext()));
            Span span = (extracted.context() != null ? getTracer().joinSpan(extracted.context()) : getTracer().nextSpan(extracted))
                        .name(spanName)
                        .kind(Span.Kind.SERVER)
                        .start();

            boolean isOneway = false, deferFinish = false;
            try(Tracer.SpanInScope scope = tracer.withSpanInScope(span)) {
                if(!span.isNoop()) {
                    span.tag("lc", "dubbo");//本地组件名
//                span.tag(Span.SPAN_PEER_SERVICE_TAG_NAME, "dubbo");//远程服务名
                    span.tag("server.isPerfTest", Boolean.toString(PerfTestContext.isCurrentInPerfTestMode()));
                    span.tag("server.thread", Thread.currentThread().getName());
                }
                Result result = invoker.invoke(invocation);
                if(result.hasException()){
                    onError(result.getException(), span);
                }
                isOneway = RpcUtils.isOneway(invoker.getUrl(), invocation);
                Future<Object> future = rpcContext.getFuture(); // the case on async client invocation
                if (!isOneway && future instanceof FutureAdapter) {
                    deferFinish = true;
                    ((FutureAdapter<Object>) future).whenComplete((v, t) -> {
                        if (t != null) {
                            onError(t, span);
                            span.finish();
                        } else {
                            span.finish();
                        }
                    });
                }
                return result;
            } catch(Exception e){
                getErrorParser().error(e, span);
                throw e;
            } finally {
                if (isOneway) {
                    span.flush();
                } else if (!deferFinish) {
                    span.finish();
                }
            }
        }
    }

    static void onError(Throwable error, Span span) {
        span.error(error);
        if (error instanceof RpcException) {
            span.tag("dubbo.error_code", Integer.toString(((RpcException) error).getCode()));
        }
    }

    static class DubboRequestTextMap {

        private final RpcContext delegate;

        DubboRequestTextMap(RpcContext delegate) {
            this.delegate = delegate;
        }

        public String get(String key) {
            return delegate.getAttachments().get(key);
        }

        public void put(String key, String value) {
            if (!StringUtils.hasText(value)) {
                return;
            }
            delegate.setAttachment(key, value);
        }
    }

}
