/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package cn.com.duibaboot.ext.autoconfigure.dubbo.dispatcher;

import cn.com.duibaboot.ext.autoconfigure.cat.context.CatConstants;
import cn.com.duibaboot.ext.autoconfigure.core.utils.CatUtils;
import com.dianping.cat.Cat;
import com.dianping.cat.message.Message;
import com.dianping.cat.message.Transaction;
import org.apache.dubbo.common.logger.Logger;
import org.apache.dubbo.common.logger.LoggerFactory;
import org.apache.dubbo.remoting.Channel;
import org.apache.dubbo.remoting.ChannelHandler;
import org.apache.dubbo.remoting.transport.dispatcher.ChannelEventRunnable;

public class DuibaChannelEventRunnable implements Runnable {
    private static final Logger logger = LoggerFactory.getLogger(DuibaChannelEventRunnable.class);

    private final ChannelHandler handler;
    private final Channel channel;
    private final ChannelEventRunnable.ChannelState state;
    private final Throwable exception;
    private final Object message;
    private final long submitNanoTime;

    public DuibaChannelEventRunnable(Channel channel, ChannelHandler handler, ChannelEventRunnable.ChannelState state) {
        this(channel, handler, state, null);
    }

    public DuibaChannelEventRunnable(Channel channel, ChannelHandler handler, ChannelEventRunnable.ChannelState state, Object message) {
        this(channel, handler, state, message, null);
    }

    public DuibaChannelEventRunnable(Channel channel, ChannelHandler handler, ChannelEventRunnable.ChannelState state, Throwable t) {
        this(channel, handler, state, null, t);
    }

    public DuibaChannelEventRunnable(Channel channel, ChannelHandler handler, ChannelEventRunnable.ChannelState state, Object message, Throwable exception) {
        this.submitNanoTime = System.nanoTime();
        this.channel = channel;
        this.handler = handler;
        this.state = state;
        this.message = message;
        this.exception = exception;
    }

    @Override
    public void run() {
        Transaction t = Cat.newTransaction(CatConstants.THREAD_POOL, CatConstants.DUBBO_PROCESSOR);
        CatUtils.newCompletedTransaction(CatConstants.THREAD_POOL, CatConstants.DUBBO_QUEUE_WAIT, submitNanoTime);
        try {
            doRun();
            t.setStatus(Message.SUCCESS);
        } finally {
            t.complete();
        }
    }

    private void doRun() {
        if (state == ChannelEventRunnable.ChannelState.RECEIVED) {
            try {
                handler.received(channel, message);
            } catch (Exception e) {
                logger.warn("ChannelEventRunnable handle " + state + " operation error, channel is " + channel
                        + ", message is " + message, e);
            }
        } else {
            Transaction t;
            switch (state) {
                case CONNECTED:
                    t = Cat.newTransaction(CatConstants.THREAD_POOL, "dubbo.connected");
                    try {
                        handler.connected(channel);
                        t.setStatus(Message.SUCCESS);
                    } catch (Exception e) {
                        logger.warn("ChannelEventRunnable handle " + state + " operation error, channel is " + channel, e);
                        t.setStatus(e.getMessage());
                    } finally {
                        t.complete();
                    }
                    break;
                case DISCONNECTED:
                    t = Cat.newTransaction(CatConstants.THREAD_POOL, "dubbo.disconnected");
                    try {
                        handler.disconnected(channel);
                        t.setStatus(Message.SUCCESS);
                    } catch (Exception e) {
                        logger.warn("ChannelEventRunnable handle " + state + " operation error, channel is " + channel, e);
                        t.setStatus(e.getMessage());
                    } finally {
                        t.complete();
                    }
                    break;
                case SENT:
                    t = Cat.newTransaction(CatConstants.THREAD_POOL, "dubbo.sent");
                    try {
                        handler.sent(channel, message);
                        t.setStatus(Message.SUCCESS);
                    } catch (Exception e) {
                        logger.warn("ChannelEventRunnable handle " + state + " operation error, channel is " + channel
                                + ", message is " + message, e);
                        t.setStatus(e.getMessage());
                    } finally {
                        t.complete();
                    }
                    break;
                case CAUGHT:
                    t = Cat.newTransaction(CatConstants.THREAD_POOL, "dubbo.caught");
                    try {
                        handler.caught(channel, exception);
                        t.setStatus(Message.SUCCESS);
                    } catch (Exception e) {
                        logger.warn("ChannelEventRunnable handle " + state + " operation error, channel is " + channel
                                + ", message is: " + message + ", exception is " + exception, e);
                        t.complete();
                    }
                    break;
                default:
                    logger.warn("unknown state: " + state + ", message is " + message);
            }
        }
    }
}
