package cn.com.duibaboot.ext.autoconfigure.dubbo.hystrix;

import com.netflix.hystrix.HystrixCommand;
import com.netflix.hystrix.HystrixCommandGroupKey;
import com.netflix.hystrix.HystrixCommandKey;
import com.netflix.hystrix.exception.HystrixBadRequestException;
import feign.MethodMetadata;
import org.apache.dubbo.common.constants.CommonConstants;
import org.apache.dubbo.common.extension.Activate;
import org.apache.dubbo.metadata.MetadataService;
import org.apache.dubbo.rpc.*;
import org.springframework.beans.BeanUtils;

/**
 * dubbo调用支持hystrix
 */
@Activate(group = {CommonConstants.CONSUMER},order = -10002)
public class DubboHystrixFilter implements Filter {

    @Override
    public Result invoke(Invoker<?> invoker, Invocation invocation) throws RpcException {
        // org.apache.dubbo.metadata.MetadataService 不包装熔断监控
        if (invoker.getInterface().equals(MetadataService.class)) {
            return invoker.invoke(invocation);
        }
        String remoteApplicationName = invoker.getUrl().getRemoteApplication();
        String methodSignature = configKey(invoker.getInterface(), invocation.getMethodName(), invocation.getParameterTypes());
        DubboHystrixCommand dubboHystrixCommand = new DubboHystrixCommand(invoker, invocation, remoteApplicationName, methodSignature);
        try {
            return dubboHystrixCommand.execute();
        } catch(HystrixBadRequestException e) {
            if(e.getCause() != null && e.getCause() instanceof RpcException) {
                throw (RpcException)e.getCause();
            }
            throw e;
        }
    }

    /**
     * Configuration keys are formatted as unresolved <a href= "http://docs.oracle.com/javase/6/docs/jdk/api/javadoc/doclet/com/sun/javadoc/SeeTag.html"
     * >see tags</a>. This method exposes that format, in case you need to create the same value as
     * {@link MethodMetadata#configKey()} for correlation purposes.
     *
     * <p>Here are some sample encodings:
     *
     * <pre>
     * <ul>
     *   <li>{@code Route53}: would match a class {@code route53.Route53}</li>
     *   <li>{@code Route53#list()}: would match a method {@code route53.Route53#list()}</li>
     *   <li>{@code Route53#listAt(Marker)}: would match a method {@code
     * route53.Route53#listAt(Marker)}</li>
     *   <li>{@code Route53#listByNameAndType(String, String)}: would match a method {@code
     * route53.Route53#listAt(String, String)}</li>
     * </ul>
     * </pre>
     *
     * Note that there is no whitespace expected in a key!
     *
     * @param targetType {@link feign.Target#type() type} of the Feign interface.
     * @see MethodMetadata#configKey()
     */
    //copy from Feign.configKey, 稍微有些不一样
    private static String configKey(Class targetType, String methodName, Class<?>[] parameterTypes) {
        StringBuilder builder = new StringBuilder();
        builder.append(targetType.getSimpleName());
        builder.append('#').append(methodName).append('(');
        for (Class param : parameterTypes) {
            builder.append(param.getSimpleName()).append(',');
        }
        if (parameterTypes.length > 0) {
            builder.deleteCharAt(builder.length() - 1);
        }
        return builder.append(')').toString();
    }

    private static class DubboHystrixCommand extends HystrixCommand<Result>{

        private RpcContext rpcContext;
        private Invoker<?> invoker;
        private Invocation invocation;

        protected DubboHystrixCommand(Invoker<?> invoker, Invocation invocation, String group, String command) {
            super(Setter.withGroupKey(HystrixCommandGroupKey.Factory.asKey(group))
                .andCommandKey(HystrixCommandKey.Factory.asKey(command)));
            this.invoker = invoker;
            this.invocation = invocation;
            this.rpcContext = RpcContext.getServiceContext();
        }

        @Override
        protected Result run() throws Exception {
            try {
                BeanUtils.copyProperties(rpcContext, RpcContext.getServiceContext());
                return invoker.invoke(invocation);
            } catch (RpcException e) {
                //包装成HystrixBadRequestException，防止触发熔断。
                throw new HystrixBadRequestException(e.getMessage(), e);
            } finally {
                BeanUtils.copyProperties(RpcContext.getServiceContext(), rpcContext);
                RpcContext.removeContext();
            }
        }

//        @Override
//        protected Result getFallback() {
////            Method method = ReflectionUtils.findMethod(invoker.getInterface(), invocation.getMethodName(), invocation.getParameterTypes());
//            AdvancedFeignClient c = invoker.getInterface().getAnnotation(AdvancedFeignClient.class);
//            if (c.fallback() != null) {
//
//            } else if(c.fallbackFactory() != null) {
//
//            }
//
//            return super.getFallback();
//        }
    }

}
