package cn.com.duibaboot.ext.autoconfigure.dubbo.hystrix;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.feign.hystrix.HystrixFeignConfiguration;
import com.netflix.hystrix.HystrixCommand;
import feign.hystrix.HystrixFeign;
import org.apache.dubbo.config.annotation.DubboService;
import org.apache.dubbo.rpc.model.ApplicationModel;
import org.apache.dubbo.rpc.model.ConsumerModel;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.cloud.openfeign.CustomFeignClientsRegistrar;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.EventListener;

import java.lang.reflect.Method;
import java.util.Collection;

/**
 * Feign客户端自定义，覆盖 FeignClientsConfiguration 中对应配置
 */
@Configuration
@ConditionalOnClass({ HystrixCommand.class, HystrixFeign.class })
public class DubboHystrixFeignConfiguration {

	private static final Logger logger = LoggerFactory.getLogger(DubboHystrixFeignConfiguration.class);

	@Configuration
	@ConditionalOnClass({DubboService.class, ApplicationModel.class})
	public static class DubboHystrixInfoConfiguration {
		@Value("${spring.application.name}")
		private String springApplicationName;

		//扫描所有dubbo reference bean，生成hystrix超时配置信息。
		@EventListener(MainContextRefreshedEvent.class)
		public void initDubboHystrixInfo() {
			Collection<ConsumerModel> consumerModels = ApplicationModel.allConsumerModels();
			if(consumerModels.isEmpty()) {
				return;
			}

			consumerModels.forEach(consumerModel -> {
				Class<?> interfaceClass = consumerModel.getServiceInterfaceClass();
				//搜寻AdvancedFeignClient类上标记的name信息
				String name = null;
				try {
					name = CustomFeignClientsRegistrar.searchAndSetClientNameFromPackageInfo(interfaceClass.getPackage().getName());
				} catch (IllegalStateException e) {
					//Ignore
				}
				if (name == null) {
					logger.warn("建议在{}类所在包上标注@AdvancedFeignClient，带上应用名，以支持在界面上配置接口超时时间。");
					return;
				}
				for (Method method : interfaceClass.getDeclaredMethods()) {
					if(method.isDefault()){
						continue;
					}
					HystrixFeignConfiguration.create(springApplicationName, name, interfaceClass, method);
				}
			});
		}
	}

}
