/*
 * Copyright (c) 2008-2017, Hazelcast, Inc. All Rights Reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package cn.com.duibaboot.ext.autoconfigure.hazelcast.eureka;

import ch.qos.logback.classic.Level;
import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.wolf.utils.NumberUtils;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.DuibaEurekaAutoServiceRegistration;
import cn.com.duibaboot.ext.autoconfigure.cloud.netflix.eureka.EurekaClientUtils;
import cn.com.duibaboot.ext.autoconfigure.hazelcast.DuibaHazelcastProperties;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Lists;
import com.hazelcast.cluster.Address;
import com.hazelcast.logging.ILogger;
import com.hazelcast.spi.discovery.AbstractDiscoveryStrategy;
import com.hazelcast.spi.discovery.DiscoveryNode;
import com.hazelcast.spi.discovery.SimpleDiscoveryNode;
import com.netflix.appinfo.ApplicationInfoManager;
import com.netflix.appinfo.InstanceInfo;
import com.netflix.discovery.EurekaClient;
import com.netflix.discovery.shared.Application;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.context.ApplicationContext;

import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

/**
 * hazelcast的集群发现策略，使用eureka来做服务发现
 * @author liuyao
 */
@Slf4j
public final class EurekaOneDiscoveryStrategy extends AbstractDiscoveryStrategy {
    static{
        ((ch.qos.logback.classic.Logger)log).setLevel(Level.INFO);
    }

    private static final Set<InstanceInfo.InstanceStatus> ENABLE_SET = ImmutableSet.of(InstanceInfo.InstanceStatus.UP,InstanceInfo.InstanceStatus.STARTING);
    private static final int NUM_RETRIES = 10;
    private static final int VERIFICATION_WAIT_TIMEOUT = 1;
    private static final int DISCOVERY_RETRY_TIMEOUT = 3;
    private static final String HAZALCAST_NODE_APP_NAME = "hazalcast.discovery.app-name";

    private final AtomicBoolean running = new AtomicBoolean(true);

    private EurekaClient eurekaClient;
    private ApplicationInfoManager applicationInfoManager;
    private DiscoveryNode currentDiscoveryNode;
    private DuibaHazelcastProperties duibaHazelcastProperties;
    private ApplicationContext applicationContext;

    protected EurekaOneDiscoveryStrategy(ILogger logger,Map<String, Comparable> properties) {
        super(logger, properties);
    }

    public void setEurekaClient(EurekaClient eurekaClient) {
        this.eurekaClient = eurekaClient;
        this.applicationInfoManager = eurekaClient.getApplicationInfoManager();
    }

    public void setDuibaHazelcastProperties(DuibaHazelcastProperties duibaHazelcastProperties) {
        this.duibaHazelcastProperties = duibaHazelcastProperties;
    }

    public void setCurrentDiscoveryNode(DiscoveryNode currentDiscoveryNode) {
        this.currentDiscoveryNode = currentDiscoveryNode;
    }

    public void setApplicationContext(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }

    private boolean isMember() {
        return currentDiscoveryNode != null;
    }

    private String getApplicationName(){
        String appName = duibaHazelcastProperties.getProperties().get(HAZALCAST_NODE_APP_NAME);
        if(StringUtils.isNotBlank(appName)){
            return appName;
        }
        return applicationInfoManager.getEurekaInstanceConfig().getAppname();
    }

    /**
     * 启动前检查是否存在节点
     */
    @Override
    public void start() {
        if(isMember()){
            //依赖一下注册器，完成本机的注册
            applicationContext.getBean(DuibaEurekaAutoServiceRegistration.class);
            //刷新本机缓存
            EurekaClientUtils.refreshRegistry(eurekaClient);
        }
    }

    @Override
    public Iterable<DiscoveryNode> discoverNodes() {
        List<DiscoveryNode> nodes = Lists.newArrayList();
        String applicationName =getApplicationName();
        Application application = null;
        for (int i = 0; i < NUM_RETRIES; i++) {
            application = eurekaClient.getApplication(applicationName);
            if (application != null) {
                break;
            }
            try {
                getLogger().info("Waiting for registration with Eureka...");
                TimeUnit.SECONDS.sleep(DISCOVERY_RETRY_TIMEOUT);
                //强制eureka从服务器重新获取一次
                EurekaClientUtils.refreshRegistry(eurekaClient);
            } catch (InterruptedException almostIgnore) {
                Thread.currentThread().interrupt();
            }
        }
        if(Objects.isNull(application)){
            log.warn("Don't find hazelcast nodes");
            return nodes;
        }
        List<InstanceInfo> instances = application.getInstancesAsIsFromEureka();
        for (InstanceInfo instance : instances) {
            // Only recognize up/starting and running instances
            if (!ENABLE_SET.contains(instance.getStatus())) {
                continue;
            }
            InetAddress address = mapAddress(instance);
            if (null == address) {
                continue;
            }
            Map<String, String> metadata = instance.getMetadata();
            String portStr = metadata.get(EurekaHazelcastDiscoveryRegister.HAZELCAST_PORT);
            String version = metadata.get(EurekaHazelcastDiscoveryRegister.HAZELCAST_VERSION);
            if(!StringUtils.equals(version,EurekaHazelcastDiscoveryRegister.getHazelcastVersion())){
                continue;
            }
            int port = NumberUtils.parseInt(portStr, -1);
            //如果metadata中没有hazelcast的端口信息，则对应服务器肯定还没有接入hazelcast，忽略该服务器.
            if(port == -1){
                continue;
            }
            nodes.add(new SimpleDiscoveryNode(new Address(address, port), metadata));
        }
        log.debug("hazelcast found nodes {}", nodes.stream().map(DiscoveryNode::getPrivateAddress).collect(Collectors.toList()));
        return nodes;
    }

    @Override
    public void destroy() {
        running.set(false);
    }

    private InetAddress mapAddress(InstanceInfo instance) {
        try {
            return InetAddress.getByName(instance.getIPAddr());
        } catch (UnknownHostException e) {
            getLogger().warning("InstanceInfo '" + instance + "' could not be resolved");
        }
        return null;
    }

}
