package cn.com.duibaboot.ext.autoconfigure.rocketmq.duiba;

import cn.com.duiba.boot.event.MainContextRefreshedEvent;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.config.ConfigurableListableBeanFactory;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.beans.factory.support.BeanDefinitionRegistryPostProcessor;
import org.springframework.beans.factory.support.GenericBeanDefinition;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.event.EventListener;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.core.env.Environment;

import java.util.Map;

/**
 * 兼容原有监听者配置方式
 * 取消监听者3个的上限限制
 * @author liuyao
 */
@Slf4j
public class DuibaRocketmqListenerContainerFactory implements BeanDefinitionRegistryPostProcessor, EnvironmentAware {

    private Environment environment;

    private BeanDefinitionRegistry registry;

    private final Map<String,ConsumerProperties> consumerPropertiesMap = Maps.newHashMap();

    private final Map<String,DuibaRocketmqListenerContainer> containers = Maps.newConcurrentMap();

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory beanFactory) throws BeansException {
        //nothing
    }

    @Override
    public void postProcessBeanDefinitionRegistry(BeanDefinitionRegistry registry) throws BeansException {
        this.registry = registry;
        DuibaRocketMqProperties duibaRocketMqProperties = Binder.get(environment).bindOrCreate("duiba.rocketmq",DuibaRocketMqProperties.class);
        registerContainer("rocketMqConsumer",null,duibaRocketMqProperties.getConsumer());
        if(duibaRocketMqProperties.getExtraConsumer()==null){
            return;
        }
        for(int i = 0;i<duibaRocketMqProperties.getExtraConsumer().length;i ++){
            String beanName = "extraRocketMqConsumer" + i;
            registerContainer(beanName,null,duibaRocketMqProperties.getConsumer());
        }
    }

    public void registerContainer(String beanName,Integer index,ConsumerProperties consumerProperties){
        if(!consumerProperties.getEnable()){
            return;
        }
        //@ConditionalOnMissingBean
        if(registry.containsBeanDefinition(beanName)){
            return;
        }
        if(consumerPropertiesMap.containsKey(consumerProperties.getGroup())){
            throw new RuntimeException("消费组["+consumerProperties.getGroup()+"]冲突，配置两个相同的消费组极易产生订阅一致性覆盖，请检查消费者配置");
        }
        consumerPropertiesMap.put(consumerProperties.getGroup(),consumerProperties);

        GenericBeanDefinition beanDefinition = new GenericBeanDefinition();
        beanDefinition.setBeanClass(DuibaRocketmqListenerContainer.class);
        beanDefinition.getPropertyValues().add("index",index);
        beanDefinition.getPropertyValues().add("consumerProperties",consumerProperties);
        beanDefinition.setScope("singleton");
        registry.registerBeanDefinition(beanName,beanDefinition);
    }

    @EventListener(MainContextRefreshedEvent.class)
    @Order(Ordered.LOWEST_PRECEDENCE)//用户应用可能要加载缓存
    public void onEvent(MainContextRefreshedEvent event){
        Map<String, DuibaRocketmqListenerContainer> map = event.getApplicationContext().getBeansOfType(DuibaRocketmqListenerContainer.class);
        if(map.isEmpty()){
            return;
        }
        for(DuibaRocketmqListenerContainer container : map.values()){
            if (!container.isRunning()) {
                try {
                    container.start();
                } catch (Exception e) {
                    log.error("Started container failed. {}", container, e);
                }
            }
            containers.put(container.getConsumerProperties().getGroup(),container);
        }
    }

    public ConsumerProperties getConsumerPropertiesByGroup(String group){
        return consumerPropertiesMap.get(group);
    }

    public DuibaRocketmqListenerContainer getContainerByGroup(String group){
        return containers.get(group);
    }

    @Override
    public void setEnvironment(Environment environment) {
        this.environment = environment;
    }

}
