package cn.com.duibaboot.ext.autoconfigure.rocketmq.duiba;

import cn.com.duibaboot.ext.autoconfigure.rocketmq.grouping.MessageListenerConcurrentlyWrapper;
import cn.com.duibaboot.ext.autoconfigure.rocketmq.grouping.MessageListenerOrderlyWrapper;
import lombok.extern.slf4j.Slf4j;
import org.apache.rocketmq.client.consumer.DefaultMQPushConsumer;
import org.apache.rocketmq.client.consumer.listener.MessageListener;
import org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently;
import org.apache.rocketmq.client.consumer.listener.MessageListenerOrderly;
import org.apache.rocketmq.client.exception.MQClientException;
import org.apache.rocketmq.spring.annotation.MessageModel;
import org.apache.rocketmq.spring.autoconfigure.RocketMQProperties;
import org.apache.rocketmq.spring.support.RocketMQListenerContainer;
import org.apache.rocketmq.spring.support.RocketMQUtil;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.NoSuchBeanDefinitionException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.SmartLifecycle;
import org.springframework.util.Assert;

import javax.annotation.Resource;
import java.util.Objects;
import java.util.Optional;

/**
 * 兼容兑吧老监听逻辑
 * @author liuyao
 */
@Slf4j
public class DuibaRocketmqListenerContainer implements InitializingBean,RocketMQListenerContainer,SmartLifecycle {

    private static final String DEFAULT = "bootRocketMqMessageListener";

    private static final String PREFIX = "extraBootRocketMqMessageListener";

    @Resource
    private ApplicationContext applicationContext;

    @Resource
    private RocketMQProperties rocketMqProperties;

    private ConsumerProperties consumerProperties;

    private Integer index;

    private String consumerGroup;

    private int consumeThreadMax = 64;

    private DefaultMQPushConsumer consumer;

    private boolean running;

    private MessageModel messageModel;

    public ConsumerProperties getConsumerProperties() {
        return consumerProperties;
    }

    public void setConsumerProperties(ConsumerProperties consumerProperties) {
        this.consumerProperties = consumerProperties;
        this.consumerGroup = consumerProperties.getGroup();
        this.consumeThreadMax = Optional.ofNullable(consumerProperties.getConsumeThreadNums()).orElse(64);
        this.messageModel = consumerProperties.getMessageModel();
    }

    public Integer getIndex() {
        return index;
    }

    public void setIndex(Integer index) {
        this.index = index;
    }

    public MessageModel getMessageModel() {
        return messageModel;
    }

    public DefaultMQPushConsumer getConsumer() {
        return consumer;
    }

    public void setConsumer(DefaultMQPushConsumer consumer) {
        this.consumer = consumer;
    }

    @Override
    public void destroy() {
        this.setRunning(false);
        if (Objects.nonNull(consumer)) {
            consumer.shutdown();
        }
        log.info("container destroyed, {}", this.toString());
    }

    @Override
    public boolean isAutoStartup() {
        return false;
    }

    @Override
    public void stop(Runnable callback) {
        stop();
        callback.run();
    }

    @Override
    public void start() {
        if (this.isRunning()) {
            throw new IllegalStateException("container already running. " + this.toString());
        }

        try {
            consumer.start();
        } catch (MQClientException e) {
            throw new IllegalStateException("Failed to start RocketMQ push consumer", e);
        }
        this.setRunning(true);
        log.info("running container: {}", this.toString());
    }

    @Override
    public void stop() {
        if (this.isRunning()) {
            if (Objects.nonNull(consumer)) {
                consumer.shutdown();
            }
            setRunning(false);
        }
    }

    @Override
    public boolean isRunning() {
        return running;
    }

    private void setRunning(boolean running) {
        this.running = running;
    }

    @Override
    public int getPhase() {
        // Returning Integer.MAX_VALUE only suggests that
        // we will be the first bean to shutdown and last bean to start
        return Integer.MAX_VALUE;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        Assert.notNull(consumerGroup, "Property 'consumerGroup' is required");
        Assert.notNull(rocketMqProperties.getNameServer(), "Property 'nameServer' is required");
        Assert.notNull(consumerProperties.getTopics(), "Property 'topic' is required");

        consumer = new DefaultMQPushConsumer(consumerGroup, false);
        consumer.setInstanceName(RocketMQUtil.getInstanceName(rocketMqProperties.getNameServer()));
        consumer.setNamesrvAddr(rocketMqProperties.getNameServer());
        consumer.setConsumeThreadMax(consumeThreadMax);
        consumer.setConsumeMessageBatchMaxSize(consumerProperties.getConsumeMessageBatchMaxSize());
        if (consumeThreadMax < consumer.getConsumeThreadMin()) {
            consumer.setConsumeThreadMin(consumeThreadMax);
        }
        int maxReconsumeTimes = consumerProperties.getMaxReconsumeTimes();
        consumer.setMaxReconsumeTimes(maxReconsumeTimes);
        switch (messageModel) {
            case BROADCASTING:
                consumer.setMessageModel(org.apache.rocketmq.common.protocol.heartbeat.MessageModel.BROADCASTING);
                break;
            case CLUSTERING:
                consumer.setMessageModel(org.apache.rocketmq.common.protocol.heartbeat.MessageModel.CLUSTERING);
                break;
            default:
                throw new IllegalArgumentException("Property 'messageModel' was wrong.");
        }
        String topics = consumerProperties.getTopics();
        String[] topicArr = topics.split(",");

        for(String topic : topicArr) {
            consumer.subscribe(topic, "*");
        }
        String messageListenerBeanId = buildBeanName(index);
        MessageListener rocketMqMessageListener;
        try {
            rocketMqMessageListener = applicationContext.getBean(messageListenerBeanId, MessageListener.class);
        }catch(NoSuchBeanDefinitionException e){
            throw new IllegalStateException("bean id:[" + messageListenerBeanId + "] class:[org.apache.rocketmq.client.consumer.listener.MessageListenerConcurrently] is not exist in spring's context, 请声明，否则不会启用rocketmq消费!!!，请参考: http://gitlab2.dui88.com/basic_platform/spring-boot-ext/wikis/Starter%E7%89%B9%E6%80%A7/_rocketmq%E3%80%90%E5%BC%80%E6%BA%90RocketMQ%E3%80%91");
        }
        //当对消息消费顺序有严格要求时，请使用MessageListenerOrderly
        if(rocketMqMessageListener instanceof MessageListenerConcurrently) {
            consumer.registerMessageListener((MessageListenerConcurrently)rocketMqMessageListener);
            if (rocketMqMessageListener instanceof MessageListenerConcurrentlyWrapper) {
                ((MessageListenerConcurrentlyWrapper)rocketMqMessageListener).setConsumerProperties(consumerProperties);
            }
        }else if(rocketMqMessageListener instanceof MessageListenerOrderly) {
            consumer.registerMessageListener((MessageListenerOrderly)rocketMqMessageListener);
            if (rocketMqMessageListener instanceof MessageListenerOrderlyWrapper) {
                ((MessageListenerOrderlyWrapper)rocketMqMessageListener).setConsumerProperties(consumerProperties);
            }
        }
    }

    public String buildBeanName(Integer index){
        if(Objects.isNull(index)){
            return DEFAULT;
        }else{
            return PREFIX + index;
        }
    }


}
