package cn.com.duibaboot.ext.stream.converter;

import org.springframework.lang.Nullable;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageHeaders;
import org.springframework.messaging.converter.AbstractMessageConverter;
import org.springframework.util.MimeType;

import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;

/**
 * A {@link org.springframework.messaging.converter.MessageConverter} to convert a
 * non-String objects to a String, when expected content type is "text/plain".
 *
 * It only performs conversions to internal format and is a wrapper around
 * {@link Object#toString()}.
 *
 * @author Marius Bogoevici
 * @author Oleg Zhurakousky
 * @since 1.2
 */
public class ObjectStringMessageConverter extends AbstractMessageConverter {

    public ObjectStringMessageConverter() {
        super(new MimeType("text", "*", Charset.forName("UTF-8")));
        setStrictContentTypeMatch(true);
    }

    protected boolean supports(Class<?> clazz) {
        return true;
    }

    @Override
    protected boolean canConvertFrom(Message<?> message, Class<?> targetClass) {
        // only supports the conversion to String
        return supportsMimeType(message.getHeaders());
    }

    @Override
    protected boolean supportsMimeType(@Nullable MessageHeaders headers) {
        MimeType mimeType = getMimeType(headers);
        if (mimeType != null) {
            for (MimeType current : getSupportedMimeTypes()) {
                if (current.getType().equals(mimeType.getType())) {
                    return true;
                }
            }
        }

        return super.supportsMimeType(headers);
    }

    protected Object convertFromInternal(Message<?> message, Class<?> targetClass,
                                         Object conversionHint) {
        if (message.getPayload() != null) {
            if (message.getPayload() instanceof byte[]) {
                if (byte[].class.isAssignableFrom(targetClass)) {
                    return message.getPayload();
                }
                else {
                    return new String((byte[]) message.getPayload(),
                            StandardCharsets.UTF_8);
                }
            }
            else {
                if (byte[].class.isAssignableFrom(targetClass)) {
                    return message.getPayload().toString()
                            .getBytes(StandardCharsets.UTF_8);
                }
                else {
                    return message.getPayload();
                }
            }
        }
        return null;
    }

    protected Object convertToInternal(Object payload, MessageHeaders headers,
                                       Object conversionHint) {
        if (payload != null) {
            if ((payload instanceof byte[])) {
                return payload;
            }
            else {
                return payload.toString().getBytes();
            }
        }
        else {
            return null;
        }
    }

}