package cn.com.duibaboot.ext.stream.converter;

import cn.com.duibaboot.ext.stream.config.BindingProperties;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.messaging.converter.*;
import org.springframework.util.CollectionUtils;
import org.springframework.util.MimeType;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * A factory for creating an instance of {@link CompositeMessageConverter} for a given
 * target MIME type.
 *
 * @author David Turanski
 * @author Ilayaperumal Gopinathan
 * @author Marius Bogoevici
 * @author Vinicius Carvalho
 * @author Oleg Zhurakousky
 */
@Slf4j
public class CompositeMessageConverterFactory {

    private final ObjectMapper objectMapper;

    private final List<MessageConverter> converters;

    public CompositeMessageConverterFactory() {
        this(Collections.<MessageConverter>emptyList(), new ObjectMapper());
    }

    /**
     * @param customConverters a list of {@link AbstractMessageConverter}
     * @param objectMapper object mapper for for serialization / deserialization
     */
    public CompositeMessageConverterFactory(List<? extends MessageConverter> customConverters, ObjectMapper objectMapper) {
        this.objectMapper = objectMapper;
        if (!CollectionUtils.isEmpty(customConverters)) {
            this.converters = new ArrayList<>(customConverters);
        }
        else {
            this.converters = new ArrayList<>();
        }
        initDefaultConverters();

        DefaultContentTypeResolver resolver = new DefaultContentTypeResolver();
        resolver.setDefaultMimeType(BindingProperties.DEFAULT_CONTENT_TYPE);
        this.converters.stream().filter(mc -> mc instanceof AbstractMessageConverter)
                .forEach(mc -> ((AbstractMessageConverter) mc)
                        .setContentTypeResolver(resolver));
    }

    private void initDefaultConverters() {
        ApplicationJsonMessageMarshallingConverter applicationJsonConverter = new ApplicationJsonMessageMarshallingConverter(this.objectMapper);
        applicationJsonConverter.setStrictContentTypeMatch(true);
        this.converters.add(applicationJsonConverter);
        this.converters.add(new ByteArrayMessageConverter() {
            @Override
            protected boolean supports(Class<?> clazz) {
                if (!super.supports(clazz)) {
                    return (Object.class == clazz);
                }
                return true;
            }
        });
        this.converters.add(new ObjectStringMessageConverter());
    }

    /**
     * Creation method.
     * @param mimeType the target MIME type
     * @return a converter for the target MIME type
     */
    public MessageConverter getMessageConverterForType(MimeType mimeType) {
        List<MessageConverter> converters = new ArrayList<>();
        for (MessageConverter converter : this.converters) {
            if (converter instanceof AbstractMessageConverter) {
                for (MimeType type : ((AbstractMessageConverter) converter)
                        .getSupportedMimeTypes()) {
                    if (type.includes(mimeType)) {
                        converters.add(converter);
                    }
                }
            }
            else {
                if (this.log.isDebugEnabled()) {
                    this.log.debug("Ommitted " + converter + " of type "
                            + converter.getClass().toString() + " for '"
                            + mimeType.toString()
                            + "' as it is not an AbstractMessageConverter");
                }
            }
        }
        if (CollectionUtils.isEmpty(converters)) {
            throw new ConversionException(
                    "No message converter is registered for " + mimeType.toString());
        }
        if (converters.size() > 1) {
            return new CompositeMessageConverter(converters);
        }
        else {
            return converters.get(0);
        }
    }

    public CompositeMessageConverter getMessageConverterForAllRegistered() {
        return new CompositeMessageConverter(new ArrayList<>(this.converters));
    }

}