package cn.com.duiba.cloud.delay.client.service;

import cn.com.duiba.cloud.delay.client.DelayMessageClient;
import cn.com.duiba.cloud.delay.client.model.DelayMessageParam;
import cn.com.duiba.cloud.delay.client.remoteservice.DelayMessageRemoteService;
import cn.hutool.core.date.DateField;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUtil;
import cn.hutool.core.util.StrUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Value;

import javax.annotation.Resource;
import java.util.Date;
import java.util.concurrent.TimeUnit;

/**
 * 延迟消息客户端impl
 *
 * @author zouweixiang
 * @date 2021/11/16
 */
@Slf4j
public class DelayMessageClientImpl implements DelayMessageClient {

    @Value("${spring.application.name}")
    private String appId;

    @Resource
    private DelayMessageRemoteService delayMessageRemoteService;

    @Value(value = "${spring.profiles.active}")
    private String env;


    @Override
    public void sendDelayMessage(String bizType, String bizNo, Integer fireTime, TimeUnit timeUnit) {
        // 计算点火时间
        long fireSeconds = timeUnit.toSeconds(fireTime);
        DateTime offset = DateUtil.offset(new Date(), DateField.SECOND, (int) fireSeconds);
        Date fireDate = new Date(offset.getTime());
        sendMessage(bizType, bizNo, fireDate);
    }


    @Override
    public void sendDelayMessageV2(String bizType, String bizNo, Date fireTime) {
        if("pre".equals(env)){
            //由于共享数据库，不共享mq的topic，因此不适用pre。或需改造
           log.warn("delay-engine don't support pre");
           return;
        }
        sendMessage(bizType, bizNo, fireTime);
    }


    @Override
    public void sendDelayMessageByApp(String appId, String bizType, String bizNo, Integer fireTime, TimeUnit timeUnit) {
        // 计算点火时间
        long fireSeconds = timeUnit.toSeconds(fireTime);
        DateTime offset = DateUtil.offset(new Date(), DateField.SECOND, (int) fireSeconds);
        Date fireDate = new Date(offset.getTime());
        sendMessageByApp(appId, bizType, bizNo, fireDate);
    }

    @Override
    public void sendDelayMessageByAppV2(String appId, String bizType, String bizNo, Date fireTime) {
        sendMessageByApp(appId, bizType, bizNo, fireTime);
    }

    /**
     * 发送消息
     *
     * @param bizType  业务类型
     * @param bizNo    业务标识
     * @param fireTime 点火时间，需要回调发送消息的时间
     */
    private void sendMessage(String bizType, String bizNo, Date fireTime) {
        sendMessageByApp(appId, bizType, bizNo, fireTime);
    }


    /**
     * 发送消息应用程序id
     *
     * @param app      应用程序
     * @param bizType  业务类型
     * @param bizNo    业务标识
     * @param fireTime 点火时间，需要回调发送消息的时间
     */
    private void sendMessageByApp(String app, String bizType, String bizNo, Date fireTime) {
        DelayMessageParam delayMessageParam = DelayMessageParam.builder()
                .appId(app)
                .bizNo(bizNo)
                .bizType(bizType)
                .fireTime(fireTime).build();
        try {
            String stringJsonResult = delayMessageRemoteService.sendDelayMessage(delayMessageParam);
            if (StrUtil.isBlank(stringJsonResult)) {
                log.warn("DelayMessageClientImpl 调用延时引擎失败:bizType:{},bizNo:{},msg:{}", bizType, bizNo, stringJsonResult);
            }
        } catch (Exception e) {
            log.error("DelayMessageClientImpl Exception 调用延时引擎失败:bizType:{},bizNo:{}", bizType, bizNo, e);
            throw e;
        }
    }
}
