package cn.com.duiba.cloud.biz.tool.utils;

import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ExecutorService;
import java.util.function.ObjIntConsumer;

/**
 * 线程池处理
 *
 * @author zhoujunquan@duiba.com.cn
 * @version 0.0.7
 * @date 2021/12/20 5:06 下午
 * @since 0.0.7
 **/
@Slf4j
public class ThreadPoolUtil {
    /**
     * 执行提交多线程操作
     *
     * @param dataList        要处理的数据集合
     * @param concurrentCount 开启的线程数
     * @param taskConsumer    实际业务处理函数 -
     *                        参数一：将总数据集合按照线程数分割的子集合；
     *                        参数二：子集合位于总集合的起始下标值
     * @param <T>             数据对象类型
     */
    public static <T> void execute(List<T> dataList, int concurrentCount, ExecutorService executorService
            , ObjIntConsumer<List<T>> taskConsumer) {
        //获取数组长度
        int taskSize = dataList.size();
        if (taskSize < concurrentCount) {
            concurrentCount = taskSize;
        }
        // 根据线程数分割，得出子数组长度
        int pageSize = (int) Math.floor((double) taskSize / concurrentCount);
        CountDownLatch latch = new CountDownLatch(concurrentCount);
        // 按线程数量进行循环操作
        for (int i = 0; i < concurrentCount; i++) {
            // 计算分割的子数组起始下标
            int start = i * pageSize;
            // 计算分割的子数组末尾下标
            int end = (i + 1) * pageSize;
            if ((i + 1) == concurrentCount) {
                end = taskSize;
            }
            // 分割子数组
            List<T> tempList = new ArrayList<>(dataList.subList(start, end));
            // 提交执行子线程
            executorService.execute(() -> {
                try {
                    taskConsumer.accept(tempList, start);
                } catch (Exception e) {
                    log.error("Thread Execute Error", e);
                } finally {
                    latch.countDown();
                }
            });
        }
        try {
            latch.await();
        } catch (InterruptedException e) {
            log.error("Latch await error", e);
            Thread.currentThread().interrupt();
        }
    }
}