package cn.com.duiba.cloud.id.generator.client.service;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.utils.NetUtils;
import cn.com.duiba.cloud.id.generator.client.configuration.Scene;
import cn.com.duiba.cloud.id.generator.client.domain.IdGeneratorRequest;
import cn.com.duiba.cloud.id.generator.client.exception.IdGeneratorException;
import cn.com.duiba.cloud.id.generator.client.remoteservice.RemoteIdGeneratorService;
import org.apache.commons.lang3.StringUtils;

import javax.annotation.Resource;
import java.util.Date;
import java.util.Objects;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;

/**
 * @author liuyao
 */
public class IdGeneratorCache implements Runnable {

    private final LinkedBlockingQueue<Long> idQueue = new LinkedBlockingQueue<>();

    private final String ip = NetUtils.getLocalIp();

    private final AtomicBoolean loading = new AtomicBoolean(false);

    private Scene scene;

    private String offsetTime;

    private Long sceneId;

    private int loadingThreshold;

    @Resource
    private RemoteIdGeneratorService remoteIdGeneratorService;
    @Resource
    private IdGeneratorAsyncLoadingService idGeneratorAsyncLoadingService;

    public void init(){
        loadingThreshold = scene.getStep() / 2;
        loadId();
    }

    public Boolean isInvalidate(){
        String time = scene.getTimeLevel().formatTime(new Date());
        return !StringUtils.equals(offsetTime,time);
    }

    public Long get() throws IdGeneratorException {
        if(Objects.equals(1,scene.getStep())){
            return getByRemote();
        }
        if(idQueue.size()<loadingThreshold && loading.compareAndSet(false,true)){
            //当发放队列保有量小于阈值时，异步进行补充
            idGeneratorAsyncLoadingService.executeLoading(this);
        }
        try {
            return idQueue.poll(1, TimeUnit.SECONDS);
        } catch (InterruptedException e) {
            throw new IdGeneratorException("发号超时");
        }
    }

    /**
     * 远程请求唯一id
     */
    private Long getByRemote() throws IdGeneratorException {
        IdGeneratorRequest request = new IdGeneratorRequest();
        request.setOffsetTime(offsetTime);
        request.setSceneId(sceneId);
        request.setIp(ip);
        try {
            return remoteIdGeneratorService.idGenerator(request);
        } catch (BizException e) {
            throw new IdGeneratorException(e.getMessage());
        }
    }

    private void loadId(){
        if(Objects.equals(1L,scene.getStep())){
            return;
        }
        Long mateId = getByRemote();
        long start = (mateId - 1) * scene.getStep();
        if(start==0){
            //从1开始发号
            start = 1L;
        }
        long end = mateId * scene.getStep();
        for(long i = start;i<end;i++){
            idQueue.add(i);
        }
        loading.set(false);
    }

    public void setScene(Scene scene) {
        this.scene = scene;
    }

    public void setOffsetTime(String offsetTime) {
        this.offsetTime = offsetTime;
    }

    public void setSceneId(Long sceneId) {
        this.sceneId = sceneId;
    }

    @Override
    public void run() {
        loadId();
    }
}
