package cn.com.duiba.cloud.id.generator.client.service;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.id.generator.client.IdGeneratorClient;
import cn.com.duiba.cloud.id.generator.client.configuration.IdGeneratorProperties;
import cn.com.duiba.cloud.id.generator.client.configuration.Scene;
import cn.com.duiba.cloud.id.generator.client.domain.IdGeneratorKey;
import cn.com.duiba.cloud.id.generator.client.domain.IdGeneratorScene;
import cn.com.duiba.cloud.id.generator.client.exception.IdGeneratorException;
import cn.com.duiba.cloud.id.generator.client.remoteservice.RemoteIdGeneratorService;
import cn.com.duiba.cloud.id.generator.client.service.builder.BizIdBuilder;
import cn.com.duiba.cloud.id.generator.client.service.builder.BizIdBuilderParams;
import cn.com.duiba.cloud.id.generator.client.service.builder.impl.DefaultBizIdBuilder;
import com.github.benmanes.caffeine.cache.CacheLoader;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import org.checkerframework.checker.nullness.qual.NonNull;
import org.springframework.beans.factory.SmartInitializingSingleton;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.ApplicationContext;
import org.springframework.scheduling.annotation.Scheduled;

import javax.annotation.Resource;
import java.util.*;

/**
 * @author liuyao
 */
public class IdGeneratorClientImpl implements SmartInitializingSingleton, IdGeneratorClient {

    @Resource
    private ApplicationContext applicationContext;
    @Resource
    private RemoteIdGeneratorService remoteIdGeneratorService;
    @Resource
    private IdGeneratorProperties idGeneratorProperties;
    @Resource
    private DefaultBizIdBuilder defaultBizIdBuilder;
    @Value("${spring.application.name}")
    private String appId;

    private final Map<String, BizIdBuilder> bizIdBuilders = Maps.newConcurrentMap();

    private final Map<String,Long> sceneIdMap = Maps.newConcurrentMap();

    private final LoadingCache<IdGeneratorKey, IdGeneratorCache> cache = Caffeine.newBuilder().build(new CacheLoader<IdGeneratorKey, IdGeneratorCache>() {
        @Override
        public IdGeneratorCache load(@NonNull IdGeneratorKey key) {
            Long sceneId = sceneIdMap.get(key.getSceneKey());
            if(Objects.isNull(sceneId)){
                throw new IdGeneratorException("发号场景["+key.getSceneKey()+"]未初始化");
            }
            // idGeneratorCache 为原型Bean
            IdGeneratorCache idGeneratorCache = applicationContext.getBean(IdGeneratorCache.class);
            idGeneratorCache.setScene(idGeneratorProperties.getScenes().get(key.getSceneKey()));
            idGeneratorCache.setOffsetTime(key.getTime());
            idGeneratorCache.setSceneId(sceneId);
            idGeneratorCache.init();
            return idGeneratorCache;
        }
    });

    /**
     * 扫描过期的发号Cache
     */
    @Scheduled(cron = "0 0 1/1 * * ?")
    public void scanCache(){
        Set<IdGeneratorKey> keys = Sets.newHashSet();
        for(Map.Entry<IdGeneratorKey,IdGeneratorCache> entry:cache.asMap().entrySet()){
            IdGeneratorCache idGeneratorCache = entry.getValue();
            if(idGeneratorCache.isInvalidate()){
                keys.add(entry.getKey());
            }
        }
        for (IdGeneratorKey key:keys){
            cache.invalidate(key);
        }
    }

    @Override
    public void afterSingletonsInstantiated(){
        //初始化发号场景
        List<IdGeneratorScene> sceneList = Lists.newArrayList();
        for(Map.Entry<String, Scene> entry:idGeneratorProperties.getScenes().entrySet()){
            Scene scene = entry.getValue();
            IdGeneratorScene item = new IdGeneratorScene();
            item.setSceneKey(entry.getKey());
            item.setAppId(appId);
            item.setStep(scene.getStep());
            sceneList.add(item);
        }
        try {
            sceneIdMap.putAll(remoteIdGeneratorService.initIdGeneratorScene(sceneList));
        } catch (BizException e) {
            throw new RuntimeException(e.getMessage());
        }
        //注册业务id构建器
        Map<String,BizIdBuilder> builderMap = applicationContext.getBeansOfType(BizIdBuilder.class);
        for(BizIdBuilder builder:builderMap.values()){
            String sceneKey = builder.sceneKey();
            if(bizIdBuilders.containsKey(sceneKey)){
                throw new RuntimeException("发号场景["+sceneKey+"]存在多个BizIdBuilder");
            }
            bizIdBuilders.put(sceneKey,builder);
        }
    }

    private Long generator(String sceneKey,Date time){
        Scene scene = idGeneratorProperties.getScenes().get(sceneKey);
        if(Objects.isNull(scene)){
            throw new IdGeneratorException("发号场景["+sceneKey+"]未配置");
        }
        String formatTime = scene.getTimeLevel().formatTime(time);
        IdGeneratorKey key = new IdGeneratorKey();
        key.setSceneKey(sceneKey);
        key.setTime(formatTime);
        IdGeneratorCache idGeneratorCache = cache.get(key);
        return Objects.requireNonNull(idGeneratorCache).get();
    }

    @Override
    public Long idGenerator(String sceneKey){
        return generator(sceneKey,new Date());
    }

    @Override
    public String bizIdGenerator(String sceneKey){
        return bizIdGenerator(sceneKey,Collections.emptyMap());
    }

    @Override
    public String bizIdGenerator(String sceneKey,Map<String,String> selfDefinedParameter){
        BizIdBuilder builder = bizIdBuilders.getOrDefault(sceneKey,defaultBizIdBuilder);
        Date time = new Date();
        Long id = generator(sceneKey,time);
        BizIdBuilderParams params = new BizIdBuilderParams();
        params.setId(id);
        params.setSelfDefinedParameter(selfDefinedParameter);
        params.setTime(time);
        return builder.build(params);
    }

}
