package cn.com.duiba.customer.link.sdk;

import cn.com.duiba.customer.link.sdk.project.ProjectInfoUtil;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.data.redis.connection.StringRedisConnection;
import org.springframework.data.redis.core.RedisCallback;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.data.redis.core.TimeoutUtils;

import java.util.List;
import java.util.concurrent.TimeUnit;

/**
 * @author 刘佳辉
 * @since 2022/11/8
 */
public class RedisApi {


    private RedisApi() {
    }

    private StringRedisTemplate stringRedisTemplate;
    private Long projectId;

    private RedisApi(Long projectId) {
        this.projectId = projectId;
        this.stringRedisTemplate = SpringBeanFactory.getBean(StringRedisTemplate.class);
    }

    public static RedisApi create() {
        return new RedisApi(ProjectInfoUtil.getAppId());
    }

    public static final String KEY_PREFIX = "customer-link:";

    private static final long TIME_OUT = 6 * 365 * 24 * 60 * 60;


    /**
     * 自定义key一定要使用此方法生成
     *
     * @param key
     * @return
     */
    public String makeKey(String key) {
        return KEY_PREFIX + projectId + ":" + key;
    }

    /**
     * 设置一个键值对,键值为{@code String}类型,并设置失效时间
     * 如果键不存在，则新增一条记录；如果键已经存在，则会修改值，并且设置失效时间
     *
     * @param key      键
     * @param value    值
     * @param l        失效时间
     * @param timeUnit 时间单位
     */
    public void set(String key, String value, long l, TimeUnit timeUnit) {
        stringRedisTemplate.opsForValue().set(makeKey(key), value, l, timeUnit);
    }

    /**
     * 通过键,获取{@code String}类型的值
     *
     * @param key 键
     * @return 值
     */
    public String get(String key) {
        return stringRedisTemplate.opsForValue().get(makeKey(key));
    }


    /**
     * 增长键对应的值，默认失效时间为6年
     * 如果键不存在，则将创建一调记录再自增，并设置默认失效时间；
     * offset可以为负数，表示自减
     *
     * @param key      键
     * @param offset   自增的值(为负数代表自减)
     * @param timeOut  超时时间, 不设置传null
     * @param timeUnit 超时时间单位, 不设置传null
     * @return 自增之后的值
     */
    public Long increment(String key, long offset, Long timeOut, TimeUnit timeUnit) {
        List<Object> pipelined = stringRedisTemplate.executePipelined((RedisCallback<Long>) connection -> {
            StringRedisConnection conn = (StringRedisConnection) connection;
            conn.incrBy(makeKey(key), offset);
            toExpire(key, timeOut, timeUnit, conn);
            return null;
        });
        return Long.parseLong(pipelined.get(0).toString());
    }

    private void toExpire(String key, Long timeOut, TimeUnit timeUnit, StringRedisConnection conn) {
        if (ObjectUtils.allNotNull(timeOut, timeUnit)) {
            long rawTimeout = TimeoutUtils.toSeconds(timeOut, timeUnit);
            conn.expire(makeKey(key), rawTimeout);
        } else {
            conn.expire(makeKey(key), TIME_OUT);
        }
    }


    /**
     * 开放stringRedisTemplate 对象
     * <p>
     * 注意！ 存储一定要按规范
     *
     * @return
     * @see RedisApi#makeKey
     */
    public StringRedisTemplate getStringRedisTemplate() {
        return stringRedisTemplate;
    }


}
