package cn.com.duiba.customer.link.sdk.http;

import cn.com.duibaboot.ext.autoconfigure.httpclient.HttpClientMethodInterceptor;
import org.apache.http.HttpResponse;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.config.Registry;
import org.apache.http.config.RegistryBuilder;
import org.apache.http.conn.socket.ConnectionSocketFactory;
import org.apache.http.conn.socket.PlainConnectionSocketFactory;
import org.apache.http.conn.ssl.NoopHostnameVerifier;
import org.apache.http.conn.ssl.SSLConnectionSocketFactory;
import org.apache.http.conn.ssl.TrustSelfSignedStrategy;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.DefaultConnectionKeepAliveStrategy;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.conn.PoolingHttpClientConnectionManager;
import org.apache.http.protocol.HttpContext;
import org.apache.http.ssl.SSLContexts;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.beans.factory.config.AbstractFactoryBean;
import org.springframework.context.ApplicationContext;
import org.springframework.core.Ordered;

import javax.net.ssl.SSLContext;
import java.security.KeyStore;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

public class HttpClientFactoryBean extends AbstractFactoryBean<CloseableHttpClient> {

    /**
     * logger
     */
    private static final Logger LOGGER = LoggerFactory.getLogger(HttpClientFactoryBean.class);

    private KeyStore keyStore;

    private ApplicationContext applicationContext;

    @Override
    protected CloseableHttpClient createInstance() {
        CloseableHttpClient pureObject = getPureObject();

        ProxyFactory proxyFactory = new ProxyFactory();
        proxyFactory.setTarget(pureObject);

        List<HttpClientMethodInterceptor> interceptors =
                new ArrayList<>(applicationContext.getBeansOfType(HttpClientMethodInterceptor.class).values());
        interceptors.sort(Comparator.comparingInt(Ordered::getOrder));
        for (HttpClientMethodInterceptor interceptor : interceptors) {
            proxyFactory.addAdvice(interceptor);
        }

        return (CloseableHttpClient) proxyFactory.getProxy();
    }

    @Override
    protected void destroyInstance(CloseableHttpClient instance) throws Exception {
        //在容器关闭前自动调用
        instance.close();
    }


    private CloseableHttpClient getPureObject() {
        HttpClientBuilder httpClientBuilder = HttpClientBuilder.create()
                .setDefaultRequestConfig(RequestConfig.custom().setConnectTimeout(5000).setSocketTimeout(5000).setConnectionRequestTimeout(10).build())
                .setMaxConnPerRoute(100)
                .setMaxConnTotal(5000)//一定要设置maxConnTotal，不然默认是10
                .setUserAgent("Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_5) AppleWebKit/537.36 (KHTML, like Gecko)" +
                        " Chrome/59.0.3071.115 Safari/537.36")//userAgent仿真，以防第三方设置的WAF判断了UA从而调用失败.
                .disableAutomaticRetries()//禁止重试
                .disableCookieManagement()
                .useSystemProperties()//for proxy
                .setKeepAliveStrategy(new DefaultConnectionKeepAliveStrategy() {
                    @Override
                    public long getKeepAliveDuration(HttpResponse response, HttpContext context) {
                        long time = super.getKeepAliveDuration(response, context);
                        if (time == -1) {
                            time = 30000;//链接最多空闲30秒
                        }
                        return time;
                    }
                })
                .evictExpiredConnections();//每隔10秒主动扫描并逐出超时的连接（超过keepAliveTimeout）
        if (keyStore == null) {
            return httpClientBuilder.build();
        }
        try {
            SSLContext sslContext =
                    SSLContexts.custom().loadTrustMaterial(keyStore, new TrustSelfSignedStrategy()).build();
            Registry<ConnectionSocketFactory> socketFactoryRegistry = RegistryBuilder.<ConnectionSocketFactory>create()
                    .register("http", PlainConnectionSocketFactory.INSTANCE)
                    .register("https", new SSLConnectionSocketFactory(sslContext, NoopHostnameVerifier.INSTANCE))
                    .build();
            PoolingHttpClientConnectionManager connManager =
                    new PoolingHttpClientConnectionManager(socketFactoryRegistry);

            //创建自定义的httpclient对象
            return httpClientBuilder
                    .setConnectionManager(connManager)
                    .setSSLSocketFactory(new SSLConnectionSocketFactory(sslContext, NoopHostnameVerifier.INSTANCE))
                    .build();
        } catch (Exception e) {
            LOGGER.error("加载Https证书失败", e);
        }
        return null;
    }

    @Override
    public Class<?> getObjectType() {
        return CloseableHttpClient.class;
    }

    @Override
    public boolean isSingleton() {
        return true;
    }

    public KeyStore getKeyStore() {
        return keyStore;
    }

    public void setKeyStore(KeyStore keyStore) {
        this.keyStore = keyStore;
    }

    public ApplicationContext getApplicationContext() {
        return applicationContext;
    }

    public void setApplicationContext(ApplicationContext applicationContext) {
        this.applicationContext = applicationContext;
    }
}
