package cn.com.duiba.customer.link.project.api.remoteservice.appisv1.utils;

import cn.com.duiba.customer.link.project.api.remoteservice.appisv1.constants.IsvConfig;
import cn.com.duiba.customer.link.project.api.remoteservice.appisv1.enums.ActStatusEnum;
import cn.com.duiba.customer.link.project.api.remoteservice.appisv1.enums.ErrorCode;
import cn.com.duiba.customer.link.project.api.remoteservice.appisv1.vo.IsvSharePrizeVO;
import cn.com.duiba.customer.link.project.api.remoteservice.appisv1.vo.IsvShareVO;
import cn.com.duiba.customer.link.project.api.remoteservice.appisv1.vo.PrizeVO;
import cn.com.duiba.projectx.sdk.BizRuntimeException;
import cn.com.duiba.projectx.sdk.StrategyResult;
import cn.com.duiba.projectx.sdk.UserRequestApi;
import cn.com.duiba.projectx.sdk.UserRequestContext;
import cn.com.duiba.projectx.sdk.component.sendprize.dto.SendPrizeResult;
import cn.com.duiba.projectx.sdk.data.ConsumerExtra;
import cn.com.duiba.projectx.sdk.data.Option;
import cn.com.duiba.projectx.sdk.utils.ConsumerApi;
import cn.com.duiba.wolf.utils.BeanUtils;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * @author lvgm
 * @date 2023/7/12
 */
public class IsvActivityUtils {

    private static final Logger LOGGER = LoggerFactory.getLogger(IsvActivityUtils.class);



    /**
     * 获取活动状态
     *
     * @param api api
     * @return ActStatusEnum
     */
    public static ActStatusEnum getActStatus(UserRequestApi api) {
        Date actStartTime = IsvOptUtils.getActStartTime(api);
        Date actEndTime = IsvOptUtils.getActEndTime(api);
        Date now = new Date();
        if (now.before(actStartTime)) {
            return ActStatusEnum.NOT_START;
        }

        if (now.after(actStartTime) && now.before(actEndTime)) {
            return ActStatusEnum.START_ING;
        }
        return ActStatusEnum.END;
    }


    /**
     * 检查活动状态并抛异常
     *
     * @param api api
     */
    public static void checkActStatusThrow(UserRequestApi api) {
        ActStatusEnum actStatus = getActStatus(api);
        if (actStatus == ActStatusEnum.NOT_START) {
            throw new BizRuntimeException(ErrorCode.ACTIVITY_NOT_START, false);
        }
        if (actStatus == ActStatusEnum.END) {
            throw new BizRuntimeException(ErrorCode.ACTIVITY_END, false);
        }
    }

    /**
     * 根据发奖规则给指定用户发奖，返回奖品信息
     * @param api
     * @param userId
     * @param ruleId
     * @return
     */
    public static PrizeVO sendPrize(UserRequestApi api,String projectName,String userId, String ruleId) {
        StrategyResult strategyResult = null;
        try {
            strategyResult = api.sendPrizeWithStrategy(userId, ruleId);
        } catch (Exception e) {
            LOGGER.error("{},发奖异常,userId=[{}],ruleId=[{}]", projectName, userId, ruleId, e);
        }
        PrizeVO prizeVO;
        if(strategyResult != null && strategyResult.getPrizeId() != null && strategyResult.getPrizeType() != null) {
            prizeVO = BeanUtils.copy(strategyResult, PrizeVO.class);
        } else {
            prizeVO = new PrizeVO();
            prizeVO.setPrizeId("thanks");
            prizeVO.setPrizeType(IsvConfig.PRIZE_TYPE_THANKS);
            prizeVO.setOptionId("thanks");
            prizeVO.setOptionName("谢谢参与");
            prizeVO.setSendCount(1L);
        }
        String prizeVOJson = JSONObject.toJSONString(prizeVO);
        LOGGER.info("{},发奖,userId=[{}],ruleId=[{}],prize=[{}]", projectName, userId, ruleId, prizeVOJson);
        return prizeVO;
    }


    /**
     * 根据发奖场景给指定用户发奖，返回奖品信息
     * @param api
     * @param sceneId
     * @return
     */
    public static PrizeVO sendPrizeBySceneId(UserRequestApi api,String projectName, String sceneId) {
        SendPrizeResult strategyResult = null;
        String userId = api.getMyUserContext().getUserId();
        try {
            strategyResult = api.getProjectApi().sendPrizeBySceneId(sceneId);
        } catch (Exception e) {
            LOGGER.error("{},发奖异常,userId=[{}],ruleId=[{}]", projectName, userId, sceneId, e);
        }
        PrizeVO prizeVO;
        if(strategyResult != null && strategyResult.getPrizeId() != null && strategyResult.getPrizeType() != null) {
            prizeVO = BeanUtils.copy(strategyResult, PrizeVO.class);
        } else {
            prizeVO = new PrizeVO();
            prizeVO.setRuId(strategyResult.getRuleId());
            prizeVO.setPrizeId("thanks");
            prizeVO.setPrizeType(IsvConfig.PRIZE_TYPE_THANKS);
            prizeVO.setOptionId("thanks");
            prizeVO.setOptionName("谢谢参与");
            prizeVO.setSendCount(1L);
        }
        String prizeVOJson = JSONObject.toJSONString(prizeVO);
        LOGGER.info("{},发奖,userId=[{}],ruleId=[{}],prize=[{}]", projectName, userId, sceneId, prizeVOJson);
        return prizeVO;
    }

    /**
     * 查直接发奖的奖品
     *
     * @param api
     * @param ruId
     * @return
     */
    public static Option queryDirectOptionByRuId(UserRequestApi api, String ruId) {
        Option option = api.getProjectApi().queryOptions(ruId).stream()
                .filter(t -> !Objects.equals(t.getPrizeType(), IsvConfig.PRIZE_TYPE_THANKS))
                .findFirst().orElseThrow(() -> new BizRuntimeException(ErrorCode.CONFIG_ERR));
        return option;
    }

    /**
     * 获取请求参数
     * @param context
     * @param name
     * @param clazz
     * @param <T>
     * @return
     */
    @SuppressWarnings("unchecked")
    public static <T> T getParameter(UserRequestContext context, String name, Class<T> clazz) {
        String value = context.getHttpRequest().getParameter(name);
        if(StringUtils.isBlank(value)) {
            return null;
        }
        try {
            if(clazz.equals(String.class)) {
                return (T) value;
            } else if(clazz.equals(Long.class)) {
                return (T) Long.valueOf(value);
            } else if(clazz.equals(Integer.class)) {
                return (T) Integer.valueOf(value);
            } else if(clazz.equals(Boolean.class)) {
                return (T) Boolean.valueOf(value);
            } else {
                return null;
            }
        } catch (Exception e) {
            return null;
        }
    }

    /**
     * 获取consumerExtra
     * @param consumerApi
     * @param userId
     * @return
     */
    public static ConsumerExtra getConsumerExtra(ConsumerApi consumerApi, String userId) {
        List<ConsumerExtra> consumerExtraList = consumerApi.listConsumerExtraByConsumerIds(Lists.newArrayList(Long.parseLong(userId)));
        if(CollectionUtils.isEmpty(consumerExtraList)) {
            return null;
        }
        return consumerExtraList.get(0);
    }


    /**
     * 分享落地页
     * @param context
     * @param api
     * @param hasPrize 是否有奖品
     * 接口文档：https://docs.dui88.com/project/1695/interface/api/141841
     * @return
     */
    public IsvShareVO coopSharePage(UserRequestContext context, UserRequestApi api,boolean hasPrize) {
        IsvShareVO data = new IsvShareVO();
        String inviteCode = context.checkAndGetStringParameter("inviteCode");
        String inviter = api.getProjectApi().getComponentApi().getInviteAssistApi().getUserIdByInviteCode(inviteCode);

        if(StringUtils.isBlank(inviter)) {
            throw new BizRuntimeException(ErrorCode.INVITER_NOT_EXIST);
        }
        ConsumerExtra consumerExtra = getConsumerExtra(api.getDuibaApi().getConsumerApi(), inviter);
        if (Objects.nonNull(consumerExtra)) {
            data.setInviterName(consumerExtra.getNickname());
            data.setInviterIcon(consumerExtra.getAvatar());
        }
        if (hasPrize){
            List<IsvSharePrizeVO> sharePrizeConfig = IsvOptUtils.getSharePrizeConfig(api);
            data.setPrizeList(sharePrizeConfig);
        }
        ActStatusEnum actStatus = getActStatus(api);
        data.setActivityStatus(actStatus.getCode());
        return data;
    }


}
