package cn.com.duiba.customer.link.project.api.remoteservice.app99349.dto;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * QQ小程序大单购活动订单信息查询响应
 * 
 * @author zhangdaqing
 * @date 2025/1/20
 */
public class QQBigBuyInfoResp {
    
    private final int returnCode;
    private final String result;
    private final List<QQBigBuyOrderInfo> orderInfoList;
    
    /**
     * 构造函数
     * @param returnCode 状态码
     * @param result 返回结果JSON字符串
     */
    public QQBigBuyInfoResp(int returnCode, String result) {
        this.returnCode = returnCode;
        this.result = result;
        this.orderInfoList = parseOrderInfoList(result);
    }
    
    /**
     * 判断是否成功
     * @return true-成功，false-失败
     */
    public boolean isSuccess() {
        return returnCode == 0;
    }
    
    /**
     * 判断是否失败
     * @return true-失败，false-成功
     */
    public boolean isFailure() {
        return returnCode != 0;
    }
    
    /**
     * 获取状态码
     * @return 状态码
     */
    public int getReturnCode() {
        return returnCode;
    }
    
    /**
     * 获取原始结果字符串
     * @return 结果字符串
     */
    public String getResult() {
        return result;
    }
    
    /**
     * 获取订单信息列表
     * @return 订单信息列表
     */
    public List<QQBigBuyOrderInfo> getOrderInfoList() {
        return orderInfoList;
    }
    
    /**
     * 获取错误信息（失败时返回）
     * @return 错误信息
     */
    public String getErrorMessage() {
        if (isFailure()) {
            return result != null ? result : "查询大单购活动订单信息失败";
        }
        return null;
    }
    
    /**
     * 解析订单信息列表
     * @param resultJson 结果JSON字符串
     * @return 订单信息列表
     */
    private List<QQBigBuyOrderInfo> parseOrderInfoList(String resultJson) {
        if (resultJson == null || resultJson.trim().isEmpty()) {
            return Collections.emptyList();
        }
        
        try {
            JSONArray jsonArray = JSON.parseArray(resultJson);
            List<QQBigBuyOrderInfo> orderList = new ArrayList<>();
            
            for (int i = 0; i < jsonArray.size(); i++) {
                JSONObject orderJson = jsonArray.getJSONObject(i);
                String orderId = orderJson.getString("OrderID");
                String giftCode = orderJson.getString("GiftCode");
                String acceptDate = orderJson.getString("AcceptDate");
                Integer lvGifts = orderJson.getInteger("LvGifts");
                
                orderList.add(new QQBigBuyOrderInfo(orderId, giftCode, acceptDate, lvGifts));
            }
            
            return orderList;
        } catch (Exception e) {
            // 解析失败时返回空列表
            return Collections.emptyList();
        }
    }
    
    /**
     * 从JSON字符串创建响应对象
     * @param jsonString JSON字符串
     * @return 响应对象
     */
    public static QQBigBuyInfoResp fromJsonString(String jsonString) {
        try {
            JSONObject json = JSONObject.parseObject(jsonString);
            int returnCode = json.getIntValue("Return");
            String result = json.getString("Result");
            return new QQBigBuyInfoResp(returnCode, result);
        } catch (Exception e) {
            throw new IllegalArgumentException("无效的响应JSON格式: " + jsonString, e);
        }
    }
    
    @Override
    public String toString() {
        return "QQBigBuyInfoResp{" +
                "returnCode=" + returnCode +
                ", result='" + result + '\'' +
                ", orderInfoList=" + orderInfoList +
                ", isSuccess=" + isSuccess() +
                '}';
    }
}
