package cn.com.duiba.developer.center.api.aspectj;

import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.developer.center.api.utils.WhiteAccessUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.core.env.Environment;
import org.springframework.stereotype.Component;
import org.springframework.util.ReflectionUtils;

import java.lang.reflect.Method;

/**
 *
 * 通过BeanPostProcessor去遍历每一个bean如果发现有
 * CustomCodePositionReport注解，就将相关方法或者类的路径上报到专门接口
 * @Author 刘凯
 * @Date 2022/8/2 9:59 上午 （可以根据需要修改）
 * @Version 1.0 （版本号）
 */
@Component
@ConditionalOnClass(WhiteAccessUtil.class)
@ConditionalOnProperty(name = "white.list.reportCodeopenSwitch",havingValue = "1",matchIfMissing = false)
public class CustomReportCollectBeanPostProccess implements BeanPostProcessor {

    private Logger logger = LoggerFactory.getLogger(CustomReportCollectBeanPostProccess.class);


    @Autowired
    private Environment environment;


    @Override
    public Object postProcessBeforeInitialization(Object bean, String beanName) throws BeansException {
        if(SpringEnvironmentUtils.isProdEnv()){
            return bean;
        }
        try {
            String sysName = environment.getProperty("spring.application.name");
            //这里还不能异步处理，不然的话，在CustomCodeReportRunner 可能会获取到空的集合
            initMethodByAnnotation(bean, sysName);
        } catch (Exception e) {
            logger.error("初始化收集CustomCodePositionReport 注解信息失败", e);
        }
        return bean;
    }

    @Override
    public Object postProcessAfterInitialization(Object bean, String beanName) throws BeansException {
        return bean;
    }


    /**
     * 在CustomReportCollectBeanPostProccess 中会扫描一轮 注解的类 还有
     * 方法上是否带有CustomCodePositionReport注解，将这些信息提前放入到whiteListCodePositionsInMethod
     * 与whiteListCodePositionInClass中
     */
    private static void initMethodByAnnotation(Object bean, String sysName) {
        // 依次获取到 使用了该注解的对象
        Class<?> aClass = AopUtils.getTargetClass(bean);

        // 获取当前对象 声明的注解 获取到注解后 还可以获取注解中的属性
        CustomCodePositionReport customCodePositionReport = aClass.getDeclaredAnnotation(CustomCodePositionReport.class);

        if (customCodePositionReport != null) {
            CustomCodeReportRunner.collectPositionObject(sysName, aClass.getName(), customCodePositionReport.description(), customCodePositionReport.value(), CustomCodeReportRunner.whiteListCodePositions);
        }


        //开始解析方法上面是否有CustomCodePositionReport注解
        Method[] methods = ReflectionUtils.getAllDeclaredMethods(bean.getClass());
        if (null != methods) {
            for (Method method : methods) {
                CustomCodePositionReport methodPositionReport = AnnotationUtils.findAnnotation(method, CustomCodePositionReport.class);
                if (null == methodPositionReport) {
                    continue;
                }

                String packagePath = aClass.getName() + "." + method.getName();
                String description = methodPositionReport.description();
                String value = methodPositionReport.value();
                CustomCodeReportRunner.collectPositionObject(sysName, packagePath, description, value, CustomCodeReportRunner.whiteListCodePositions);
            }
        }
    }


}




