package cn.com.duiba.developer.center.api.utils;

import cn.com.duiba.boot.utils.SpringEnvironmentUtils;
import cn.com.duiba.developer.center.api.aspectj.CustomCodePositionReport;
import cn.com.duiba.developer.center.api.domain.dto.authority.BusinessWhiteListContentDto;
import cn.com.duiba.developer.center.api.domain.dto.authority.BusinessWhiteListDto;
import cn.com.duiba.developer.center.api.domain.enums.authority.AccessStatusType;
import cn.com.duiba.developer.center.api.domain.enums.authority.BusinessWhiteListType;
import cn.com.duiba.developer.center.api.remoteservice.authority.RemoteBusinessWhiteListContentService;
import cn.com.duiba.developer.center.api.remoteservice.authority.RemoteBusinessWhiteListService;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.github.benmanes.caffeine.cache.Cache;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.google.common.collect.Lists;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.lang.reflect.Method;
import java.util.List;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * Created by liukai on 2020/12/27.
 * 该工具主要用户封装常用app 应用白名单的操作
 */
@Service("whiteAccessUtil")
public class WhiteAccessUtilBase implements InitializingBean {


    private static Logger logger = LoggerFactory.getLogger(WhiteAccessUtilBase.class);

    @Autowired
    private RemoteBusinessWhiteListContentService remoteBusinessWhiteListContentService;
    @Autowired
    private RemoteBusinessWhiteListService remoteBusinessWhiteListService;
    @Resource(name = "stringRedisTemplate")
    private StringRedisTemplate stringRedisTemplate;


    private static String EXITST = "1";

    private static String NOT_EXITST = "0";

    public static String PREFIX = "WHITE_LIST_";

    public static String PREFIX_SINGLE = "WHITE_LIST_SINGLE";

    public static String PREFIX_JSON = "WHITE_LIST_JSON";


    private static final Cache<String, Boolean> CAN_ACCESS_WHITE_LIST = Caffeine.newBuilder().expireAfterWrite(3, TimeUnit.MINUTES).maximumSize(1000)
            .build();

    private static final Cache<String, String> WHITE_LIST_JSON_VALUE = Caffeine.newBuilder().expireAfterWrite(3, TimeUnit.MINUTES).maximumSize(1000)
            .build();

    private static final Cache<String, List<String>> WHITE_LIST_VALUES = Caffeine.newBuilder().expireAfterWrite(3, TimeUnit.MINUTES).maximumSize(1000)
            .build();


    @Override
    public void afterPropertiesSet() throws Exception {
        ProxyFactory proxyFactory = new ProxyFactory();
        proxyFactory.setTarget(this);
        proxyFactory.addAdvice(new WhiteAccessMethodInterceptor());
        WhiteAccessUtil.whiteAccessUtilBase = (WhiteAccessUtilBase) proxyFactory.getProxy();
        logger.info("WhiteAccessUtilBase 代理设置完毕");
    }


    /**
     *  需要在堆栈中找到具体执行的方法，我们需要取到使用WhiteAccessUtil执行的类/方法，cn.com.duiba.developer.center.api.utils.WhiteAccessUtil的下一个
     * 	"className": "cn.com.duiba.developer.center.api.utils.WhiteAccessUtil",
     * 	"fileName": "WhiteAccessUtil.java",
     * 	"lineNumber": 25,
     * 	"methodName": "selectWhiteListConfig",
     * 	"nativeMethod": false
     * }, {
     * 	"className": "cn.com.duiba.home.biz.WhiteAccessTest",
     * 	"fileName": "WhiteAccessTest.java",
     * 	"lineNumber": 38,
     * 	"methodName": "selectAppItemList",
     * 	"nativeMethod": false
     * }
     */
    private static class WhiteAccessMethodInterceptor implements MethodInterceptor {

        @Override
        public Object invoke(MethodInvocation invocation) throws Throwable {
            if (SpringEnvironmentUtils.isProdEnv()) {
                return invocation.proceed();
            }

            //线上减少反射里面判断的开销
            final StackTraceElement[] stackTrace = Thread.currentThread().getStackTrace();
            StackTraceElement stackTraceElementCur = null;
            for (int i = 0; i < stackTrace.length; i++) {
                if(stackTrace[i].getClassName().equals("cn.com.duiba.developer.center.api.utils.WhiteAccessUtil") && stackTrace[i].getFileName().equals("WhiteAccessUtil.java")){
                    //这里取下一个，才是真正执行的方法
                    stackTraceElementCur = stackTrace[i+1];
                    break;
                }
            }

            String methodName = stackTraceElementCur.getMethodName();
            String className = stackTraceElementCur.getClassName();
            final Class<?> aClass = Class.forName(className);

//             获取当前对象 声明的注解 获取到注解后 还可以获取注解中的属性
            CustomCodePositionReport classCustomCodePositionReport = aClass.getDeclaredAnnotation(CustomCodePositionReport.class);

            if (classCustomCodePositionReport != null) {
                return invocation.proceed();
            }
            final Method[] declaredMethods = aClass.getDeclaredMethods();
            for (Method declaredMethod : declaredMethods) {
                if (declaredMethod.getName().equals(methodName)) {
                    //判断当前类所在方法或者类上面有没有 CustomCodePositionReport 注解，如果没有强制启动报错
                    CustomCodePositionReport methodPositionReport = AnnotationUtils.findAnnotation(declaredMethod, CustomCodePositionReport.class);

                    if (methodPositionReport != null) {
                        return invocation.proceed();
                    }
                }
            }

            throw new RuntimeException("WhiteAccessUtil 所在类或者方法需要配置CustomCodePositionReport一起使用！");
        }
    }


    /**
     * 在某个业务里面 获取对应配置的json信息
     * 只能查询json 类型的配置
     *
     * @param uniqueCode
     * @return
     */
    public String selectWhiteListJsonConfig(String uniqueCode) {
        return WHITE_LIST_JSON_VALUE.get(uniqueCode, code -> {
            String uniqueCodeKey = PREFIX_JSON + code;
            try {
                String jsonValue = stringRedisTemplate.opsForValue().get(uniqueCodeKey);
                if (jsonValue == null) {
                    BusinessWhiteListDto whiteListConfig = remoteBusinessWhiteListService.getByCode(code);

                    if (whiteListConfig == null || !BusinessWhiteListType.JSON_VALUE.getCode().equals(whiteListConfig.getBizType())) {
                        logger.warn("WHITE_LIST_JSON_VALUE  错误 请核对 queryParam = {}", code);
                        return null;
                    }

                    BusinessWhiteListContentDto dto = remoteBusinessWhiteListContentService.getByRelId(whiteListConfig.getId());
                    if (dto != null && StringUtils.isNotBlank(dto.getBizContent())) {
                        stringRedisTemplate.opsForValue().set(uniqueCodeKey, dto.getBizContent());
                        stringRedisTemplate.expire(uniqueCode, 12, TimeUnit.HOURS);
                        return dto.getBizContent();
                    } else {
                        return null;
                    }
                }
                return jsonValue;
            } catch (Exception e) {
                logger.warn("白名单查询失败。走默认降级策略 false equryParam = {}", code, e);
                return null;
            }
        });
    }


    /**
     * 判断这个appid 是否在这个业务名单里面
     *
     * @param appId
     * @param uniqueCode
     * @return json
     */
    public Boolean matchWhiteList(Long appId, String uniqueCode) {
        String appIdCode = appId.toString() + "#" + uniqueCode;
        return CAN_ACCESS_WHITE_LIST.get(appIdCode, appIdCodeStr -> {
            String[] data = appIdCodeStr.split("#");
            String code = data[1];
            String appIdStr = data[0];
            String uniqueCodeKey = PREFIX_SINGLE + code;
            //1 先查redis 判断是否存在，无论是否在白名单内，都会有值 除非过期了。需要到mysql 查询
            try {
                Object score = stringRedisTemplate.opsForHash().get(uniqueCodeKey, appIdStr);
                if (score == null) {
                    BusinessWhiteListDto whiteListConfig = remoteBusinessWhiteListService.getByCode(code);
                    if (whiteListConfig == null || AccessStatusType.CLOSE.getCode().equals(whiteListConfig.getOpenStatus()) || BusinessWhiteListType.JSON_VALUE.getCode().equals(whiteListConfig.getBizType())) {
                        logger.info("白名单类型查询错误，请核对 queryParam = {}", appIdCodeStr);
                        stringRedisTemplate.opsForHash().put(uniqueCodeKey, appIdStr, NOT_EXITST);
                        return false;
                    }
                    Boolean wasExist = remoteBusinessWhiteListContentService.getByRelIdAndValue(whiteListConfig.getId(), appId) != null;
                    stringRedisTemplate.opsForHash().put(uniqueCodeKey, appIdStr, (wasExist ? EXITST : NOT_EXITST));
                    stringRedisTemplate.expire(uniqueCodeKey, 12, TimeUnit.HOURS);
                    return wasExist;
                }
                return EXITST.equals(String.valueOf(score));
            } catch (Exception e) {
                logger.warn("白名单查询失败。走默认降级策略 false equryParam = {}", appIdCodeStr, e);
                return false;
            }
        });
    }


    /**
     * 在某个业务里面  所有名单内容
     * 只能查询白名单/黑名单类型
     *
     * @param uniqueCode
     * @return json
     */
    public List<String> selectWhiteListConfig(String uniqueCode) {
        return WHITE_LIST_VALUES.get(uniqueCode, code -> {
            String uniqueCodeKey = PREFIX + code;
            //1 先查redis 判断是否存在，无论是否在白名单内，都会有值 除非过期了。需要到mysql 查询
            try {
                String jsonValue = stringRedisTemplate.opsForValue().get(uniqueCodeKey);
                if (jsonValue == null) {
                    List<String> values = Lists.newArrayList();
                    BusinessWhiteListDto whiteListConfig = remoteBusinessWhiteListService.getByCode(code);
                    if (whiteListConfig == null) {
                        logger.warn("fetchWhiteListValues code 错误 请核对 queryParam = {}", code);
                        return values;
                    }

                    if (BusinessWhiteListType.JSON_VALUE.getCode().equals(whiteListConfig.getBizType())) {
                        logger.warn("WHITE_LIST_JSON_VALUE 白名单json类型错误，请核对 queryParam = {}", code);
                        return values;
                    }
                    List<BusinessWhiteListContentDto> dtos = remoteBusinessWhiteListContentService.getByRid(whiteListConfig.getId());
                    if (CollectionUtils.isNotEmpty(dtos)) {
                        values.addAll(dtos.stream().map(a -> a.getRelValue()).collect(Collectors.toList()));
                        stringRedisTemplate.opsForValue().set(uniqueCodeKey, JSON.toJSONString(values));
                        stringRedisTemplate.expire(uniqueCodeKey, 12, TimeUnit.HOURS);
                    }
                    return values;
                }
                return JSONArray.parseArray(jsonValue, String.class);
            } catch (Exception e) {
                logger.warn("白名单查询失败。走默认降级策略 false equryParam = {}", code, e);
                return Lists.newArrayList();
            }
        });
    }


    public static class QueryParam {

        /**
         * 需要判断的appid
         */
        private Long appId;

        /**
         * 判断appId 是否在uniqueCode 所在的白名单内
         */
        private String uniqueCode;

        public Long getAppId() {
            return appId;
        }

        public void setAppId(Long appId) {
            this.appId = appId;
        }

        public String getUniqueCode() {
            return uniqueCode;
        }

        public void setUniqueCode(String uniqueCode) {
            this.uniqueCode = uniqueCode;
        }

        public QueryParam(Long appId, String uniqueCode) {
            this.appId = appId;
            this.uniqueCode = uniqueCode;
        }

        public QueryParam(String uniqueCode) {
            this.uniqueCode = uniqueCode;
        }
    }


}
