package cn.com.duiba.api.tools;

import cn.com.duiba.api.enums.ActivityToolTypeEnum;
import com.google.common.collect.ImmutableSet;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;

import java.util.List;
import java.util.stream.Collectors;

/**
 * @Author: lufeng
 * @Description:活动工具 工具类
 * @Date: Created in 2018/11/6
 */
public class ActivityToolTypeUtil {

    private ActivityToolTypeUtil(){
        //空构造器
    }

    //duiba_hdtool表内的subType字段值基本是'xxx_xx'格式。类型名称可根据前面的字符确认。
    private static final String FORMAT = "_";

    private static final ImmutableSet<Integer> typeSet = ImmutableSet.of(1, 2, 3, 4, 5, 6);

    private static final ImmutableSet<Integer> operatingTypeSet = ImmutableSet.of(6, 7, 8, 9, 11, 12, 101);

    //通过subType获取活动类型名称
    public static String getTypeNameBySubType(String subType) {
        if (StringUtils.isEmpty(subType)) {
            return "";
        }
        String newSubType = subType.split(FORMAT)[0];
        ActivityToolTypeEnum typeEnum = ActivityToolTypeEnum.getBySubType(newSubType);
        return null == typeEnum ? "" : typeEnum.getName();
    }

    //通过type获取活动类型名称.如果结果集有多个，则只返回最大类的一个。
    public static String getTypeNameByType(Integer type) {
        String typeName = "";
        if (!typeSet.contains(type)) {
            return typeName;
        }
        //摇奖机类型存在多个type同一名称情况
        if (ActivityToolTypeEnum.TIGER.getType().equals(type)) {
            return ActivityToolTypeEnum.TIGER.getName();
        }
        //自定义类型存在多个type，只取大类自定义
        if (ActivityToolTypeEnum.CUSTOM.getType().equals(type)) {
            return ActivityToolTypeEnum.CUSTOM.getName();
        }
        List<ActivityToolTypeEnum> typeEnums = ActivityToolTypeEnum.getByType(type);
        if (CollectionUtils.isEmpty(typeEnums)) {
            return typeName;
        }
        List<String> typeNames = typeEnums.stream().filter(typeEnum -> typeEnum.getType().equals(type))
                .map(ActivityToolTypeEnum::getName)
                .collect(Collectors.toList());
        return typeNames.get(0);
    }

    //通过operatingType获取活动类型名称.如果结果集有多个，则只返回最大类的一个。
    public static String getTypeNameByOperatingType(Integer operatingType) {
        String typeName = "";
        if (!operatingTypeSet.contains(operatingType)) {
            return typeName;
        }
        //摇奖机类型存在多个type同一名称情况
        if (ActivityToolTypeEnum.TIGER.getOperatingType().equals(operatingType)) {
            return ActivityToolTypeEnum.TIGER.getName();
        }
        //自定义类型存在多个type，只取大类自定义
        if (ActivityToolTypeEnum.CUSTOM.getOperatingType().equals(operatingType)) {
            return ActivityToolTypeEnum.CUSTOM.getName();
        }

        List<ActivityToolTypeEnum> typeEnums = ActivityToolTypeEnum.getByOperatingType(operatingType);
        if (CollectionUtils.isEmpty(typeEnums)) {
            return typeName;
        }
        return typeEnums.get(0).getName();
    }

    /**
     * 根据入库活动的type和子类型查询活动类型名称
     * 兼容针对开发者自有的活动返回返回具体子类型而非大类类型
     * 注意区分该方法与getTypeNameByOperatingType的区别
     * 如果传入的customSubType为空，则默认返回第一个类型
     * @param operatingType
     * @param customSubType
     * @return
     */
    public static String  getNameByOpTypeAndCustomType(Integer operatingType,String customSubType){
        String typeName = "";
        if (!operatingTypeSet.contains(operatingType)) {
            return typeName;
        }
        //摇奖机类型存在多个type同一名称情况
        if (ActivityToolTypeEnum.TIGER.getOperatingType().equals(operatingType)) {
            return ActivityToolTypeEnum.TIGER.getName();
        }
        List<ActivityToolTypeEnum> typeEnums = ActivityToolTypeEnum.getByOperatingType(operatingType);
        if (CollectionUtils.isEmpty(typeEnums)) {
            return typeName;
        }
        //如果匹配到的长度为1，则直接返回
        if(typeEnums.size() == 1){
            return typeEnums.get(0).getName();
        }
        if(StringUtils.isBlank(customSubType)){
            return typeEnums.get(0).getName();
        }
        List<String> typeNames = typeEnums.stream().filter(typeEnum -> typeEnum.getCustomSubType().equals(customSubType))
                .map(ActivityToolTypeEnum::getName)
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(typeNames)){
            return typeName;
        }
        return typeNames.get(0);
    }
}
