package cn.com.duiba.cat.util;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

public class Scanners {

    public Scanners() {
    }

    public static Scanners.DirScanner forDir() {
        return Scanners.DirScanner.INSTANCE;
    }

    public static Scanners.JarScanner forJar() {
        return Scanners.JarScanner.INSTANCE;
    }

    public static enum JarScanner {
        INSTANCE;

        private JarScanner() {
        }

        public ZipEntry getEntry(String jarFileName, String name) {
            ZipFile zipFile = null;

            try {
                zipFile = new ZipFile(jarFileName);
                ZipEntry entry = zipFile.getEntry(name);
                ZipEntry var5 = entry;
                return var5;
            } catch (IOException var15) {
            } finally {
                if (zipFile != null) {
                    try {
                        zipFile.close();
                    } catch (IOException var14) {
                    }
                }

            }

            return null;
        }

        public byte[] getEntryContent(String jarFileName, String entryPath) {
            byte[] bytes = null;
            ZipFile zipFile = null;

            try {
                zipFile = new ZipFile(jarFileName);
                ZipEntry entry = zipFile.getEntry(entryPath);
                if (entry != null) {
                    InputStream inputStream = zipFile.getInputStream(entry);
                    bytes = Files.forIO().readFrom(inputStream);
                }
            } catch (Exception var15) {
            } finally {
                if (zipFile != null) {
                    try {
                        zipFile.close();
                    } catch (Exception var14) {
                    }
                }

            }

            return bytes;
        }

        public boolean hasEntry(String jarFileName, String name) {
            return this.getEntry(jarFileName, name) != null;
        }

        public List<String> scan(File base, Scanners.IMatcher<File> matcher) {
            List<String> files = new ArrayList();
            this.scanForFiles(base, matcher, false, files);
            return files;
        }

        public List<String> scan(ZipFile zipFile, Scanners.IMatcher<ZipEntry> matcher) {
            List<String> files = new ArrayList();
            this.scanForEntries(zipFile, matcher, false, files);
            return files;
        }

        private void scanForEntries(ZipFile zipFile, Scanners.IMatcher<ZipEntry> matcher, boolean foundFirst, List<String> names) {
            Enumeration entries = zipFile.entries();

            while (entries.hasMoreElements()) {
                ZipEntry entry = (ZipEntry) entries.nextElement();
                String name = entry.getName();
                Scanners.IMatcher.Direction direction;
                if (matcher.isDirEligible() && entry.isDirectory()) {
                    direction = matcher.matches(entry, name);
                    if (direction.isMatched()) {
                        names.add(name);
                    }
                } else if (matcher.isFileElegible() && !entry.isDirectory()) {
                    direction = matcher.matches(entry, name);
                    if (direction.isMatched()) {
                        names.add(name);
                    }
                }

                if (foundFirst && names.size() > 0) {
                    break;
                }
            }

        }

        private void scanForFiles(File jarFile, Scanners.IMatcher<File> matcher, boolean foundFirst, List<String> names) {
            ZipFile zipFile = null;

            try {
                zipFile = new ZipFile(jarFile);
            } catch (IOException var10) {
            }

            if (zipFile != null) {
                Enumeration entries = zipFile.entries();

                while (entries.hasMoreElements()) {
                    ZipEntry entry = (ZipEntry) entries.nextElement();
                    String name = entry.getName();
                    Scanners.IMatcher.Direction direction;
                    if (matcher.isDirEligible() && entry.isDirectory()) {
                        direction = matcher.matches(jarFile, name);
                        if (direction.isMatched()) {
                            names.add(name);
                        }
                    } else if (matcher.isFileElegible() && !entry.isDirectory()) {
                        direction = matcher.matches(jarFile, name);
                        if (direction.isMatched()) {
                            names.add(name);
                        }
                    }

                    if (foundFirst && names.size() > 0) {
                        break;
                    }
                }
            }

        }

        public String scanForOne(File jarFile, Scanners.IMatcher<File> matcher) {
            List<String> files = new ArrayList(1);
            this.scanForFiles(jarFile, matcher, true, files);
            return files.isEmpty() ? null : (String) files.get(0);
        }
    }

    public static enum DirScanner {
        INSTANCE;

        private DirScanner() {
        }

        public List<File> scan(File base, Scanners.IMatcher<File> matcher) {
            List<File> files = new ArrayList();
            StringBuilder relativePath = new StringBuilder();
            this.scanForFiles(base, relativePath, matcher, false, files);
            return files;
        }

        private void scanForFiles(File base, StringBuilder relativePath, Scanners.IMatcher<File> matcher, boolean foundFirst, List<File> files) {
            int len = relativePath.length();
            File dir = len == 0 ? base : new File(base, relativePath.toString());
            String[] list = dir.list();
            if (list != null) {
                String[] var9 = list;
                int var10 = list.length;

                for (int var11 = 0; var11 < var10; ++var11) {
                    String item = var9[var11];
                    File child = new File(dir, item);
                    if (len > 0) {
                        relativePath.append('/');
                    }

                    relativePath.append(item);
                    Scanners.IMatcher.Direction direction;
                    if (matcher.isDirEligible() && child.isDirectory()) {
                        direction = matcher.matches(base, relativePath.toString());
                        switch (direction) {
                            case MATCHED:
                                files.add(child);
                                break;
                            case DOWN:
                                this.scanForFiles(base, relativePath, matcher, foundFirst, files);
                        }
                    } else if (matcher.isFileElegible()) {
                        direction = matcher.matches(base, relativePath.toString());
                        switch (direction) {
                            case MATCHED:
                                if (child.isFile()) {
                                    files.add(child);
                                }
                                break;
                            case DOWN:
                                if (child.isDirectory()) {
                                    this.scanForFiles(base, relativePath, matcher, foundFirst, files);
                                }
                        }
                    }

                    relativePath.setLength(len);
                    if (foundFirst && files.size() > 0) {
                        break;
                    }
                }
            }

        }

        public File scanForOne(File base, Scanners.IMatcher<File> matcher) {
            List<File> files = new ArrayList(1);
            StringBuilder relativePath = new StringBuilder();
            this.scanForFiles(base, relativePath, matcher, true, files);
            return files.isEmpty() ? null : (File) files.get(0);
        }
    }

    public interface IMatcher<T> {

        boolean isDirEligible();

        boolean isFileElegible();

        Scanners.IMatcher.Direction matches(T var1, String var2);

        public static enum Direction {
            MATCHED,
            DOWN,
            NEXT;

            private Direction() {
            }

            public boolean isDown() {
                return this == DOWN;
            }

            public boolean isMatched() {
                return this == MATCHED;
            }

            public boolean isNext() {
                return this == NEXT;
            }
        }
    }

    public abstract static class ZipEntryMatcher implements Scanners.IMatcher<ZipEntry> {

        public ZipEntryMatcher() {
        }

        public boolean isDirEligible() {
            return false;
        }

        public boolean isFileElegible() {
            return true;
        }

        public abstract Scanners.IMatcher.Direction matches(ZipEntry var1, String var2);
    }

    public abstract static class FileMatcher implements Scanners.IMatcher<File> {

        public FileMatcher() {
        }

        public boolean isDirEligible() {
            return false;
        }

        public boolean isFileElegible() {
            return true;
        }
    }

    public abstract static class DirMatcher implements Scanners.IMatcher<File> {

        public DirMatcher() {
        }

        public boolean isDirEligible() {
            return true;
        }

        public boolean isFileElegible() {
            return false;
        }
    }
}
