package cn.com.duiba.cat.model.configuration;

import cn.com.duiba.cat.Cat;
import cn.com.duiba.cat.analyzer.MetricTagAggregator;
import cn.com.duiba.cat.message.spi.MessageTree;
import cn.com.duiba.cat.model.configuration.client.entity.ClientConfig;
import cn.com.duiba.cat.model.configuration.client.entity.Server;
import cn.com.duiba.cat.model.configuration.property.entity.Property;
import cn.com.duiba.cat.model.configuration.property.entity.PropertyConfig;
import cn.com.duiba.cat.model.configuration.property.transform.DefaultSaxParser;
import cn.com.duiba.cat.util.NetworkHelper;
import cn.com.duiba.cat.util.Properties;
import cn.com.duiba.cat.util.Splitters;
import cn.com.duiba.cat.util.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.UnsupportedEncodingException;
import java.net.URL;
import java.net.URLEncoder;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

public class DefaultClientConfigService implements ClientConfigService {

    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultClientConfigService.class);

    private static   DefaultClientConfigService instance     = new DefaultClientConfigService();
    private          ClientConfig               config;
    private          String                     routers;
    private volatile double                     samplingRate = 1d;
    private volatile boolean                    block        = false;
    private volatile int                        timeout      = 1000;
    private          MessageTreeTypeParser      treeParser   = new MessageTreeTypeParser();
    private          Map<String, List<Integer>> longConfigs  = new LinkedHashMap<String, List<Integer>>();

    private DefaultClientConfigService() {
        String config = System.getProperty(Cat.CLIENT_CONFIG);

        if (StringUtils.isNotEmpty(config)) {
            try {
                this.config = cn.com.duiba.cat.model.configuration.client.transform.DefaultSaxParser.parse(config);
                LOGGER.info("setup cat with config:" + config);
            } catch (Exception e) {
                LOGGER.error("error in client config " + config, e);
            }
        }

        if (this.config == null) {
            String appName = ApplicationEnvironment.loadAppName(Cat.UNKNOWN);
            ClientConfig defaultConfig = ApplicationEnvironment.loadClientConfig(appName);

            defaultConfig.setDomain(appName);
            this.config = defaultConfig;
            LOGGER.info("setup cat with default configuration:" + this.config);
        }
    }

    public static DefaultClientConfigService getInstance() {
        return instance;
    }

    @Override
    public int getClientConnectTimeout() {
        return timeout;
    }

    @Override
    public String getDomain() {
        return config.getDomain();
    }

    @Override
    public int getLongConfigThreshold(String key) {
        List<Integer> values = longConfigs.get(key);
        int value;

        if (values != null && !values.isEmpty()) {
            value = values.get(0);
        } else {
            value = ProblemLongType.findByName(key).getThreshold();
        }

        return value;
    }

    @Override
    public int getLongThresholdByDuration(String key, int duration) {
        List<Integer> values = longConfigs.get(key);

        if (values != null) {
            for (int i = values.size() - 1; i >= 0; i--) {
                int userThreshold = values.get(i);

                if (duration >= userThreshold) {
                    return userThreshold;
                }
            }
        }

        return -1;
    }

    @Override
    public String getRouters() {
        if (routers == null) {
            refreshConfig();
        }
        return routers;
    }

    public double getSamplingRate() {
        return samplingRate;
    }

    private String getServerConfigUrl(ClientConfig config, int index) {
        List<Server> servers = config.getServers();
        int size = servers.size();
        Server server = servers.get(index % size);
        int httpPort = server.getHttpPort();
        String serverIp = server.getIp();

        String hostname = NetworkInterfaceManager.INSTANCE.getLocalHostName();
        String ip = NetworkInterfaceManager.INSTANCE.getLocalHostAddress();

        try {
            hostname = URLEncoder.encode(hostname, "utf-8");
        } catch (UnsupportedEncodingException ignored) {
        }

        return String.format("http://%s:%d/cat/s/router?domain=%s&ip=%s&op=xml&env=%s&hostname=%s", serverIp.trim(), httpPort, getDomain(), ip, ApplicationEnvironment.ENVIRONMENT, hostname);
    }

    @Override
    public List<Server> getServers() {
        return config.getServers();
    }

    private boolean isDevMode() {
        String devMode = Properties.forString().fromEnv().fromSystem().getProperty("devMode", "false");

        return "true".equals(devMode);
    }

    public boolean isMessageBlock() {
        return block;
    }

    @Override
    public MessageType parseMessageType(MessageTree tree) {
        if (!tree.canDiscard()) {
            return MessageType.NORMAL_MESSAGE;
        } else {
            return treeParser.parseMessageType(tree);
        }
    }

    public void refreshConfig() {
        int retry = 0;
        int start = (int) (Math.random() * 10);
        int maxRetryCount = 3;
        boolean refreshStatus = false;

        while (retry < maxRetryCount) {
            String url = getServerConfigUrl(config, start + retry);

            try {
                refreshConfig(url);
                refreshStatus = true;
                break;
            } catch (Exception e) {
                retry++;
                LOGGER.error("error when connect cat server config url " + url);
            }
        }

        if ((!refreshStatus) && (!isDevMode())) {
            try {
                String xml = ApplicationEnvironment.loadRemoteClientConfig();
                ClientConfig config = cn.com.duiba.cat.model.configuration.client.transform.DefaultSaxParser.parse(xml);

                config.setDomain(getDomain());

                String url = getServerConfigUrl(config, start);

                refreshConfig(url);
            } catch (Exception e) {
                LOGGER.error("error when connect cat server config url from remote config");
            }
        }
    }

    @Override
    public void refreshConfig(PropertyConfig routerConfig) {
        refreshRouters(routerConfig);
        refreshInnerConfig(routerConfig);
    }

    private void refreshConfig(String url) throws Exception {
        String content = NetworkHelper.readFromUrlWithRetry(url);
        PropertyConfig routerConfig = DefaultSaxParser.parse(content.trim());

        if (refreshRouters(routerConfig)) {
            storeServersByUrl(url);
            refreshInnerConfig(routerConfig);
        }
    }

    private void refreshInnerConfig(PropertyConfig routerConfig) {
        samplingRate = Double.parseDouble(routerConfig.findProperty("sample").getValue());

        if (samplingRate <= 0) {
            samplingRate = 0;
        }

        block = Boolean.parseBoolean(routerConfig.findProperty("block").getValue());

        if (block) {
            Cat.disable();
        } else {
            Cat.enable();
        }

        String multiInstancesConfig = routerConfig.findProperty("multiInstances").getValue();
        if (StringUtils.isNotEmpty(multiInstancesConfig)) {
            boolean multiInstances = Boolean.parseBoolean(multiInstancesConfig);

            if (multiInstances) {
                Cat.enableMultiInstances();
            } else {
                Cat.disableMultiInstances();
            }
        }

        String startTypes = routerConfig.findProperty("startTransactionTypes").getValue();
        String matchTypes = routerConfig.findProperty("matchTransactionTypes").getValue();

        for (ProblemLongType longType : ProblemLongType.values()) {
            final String name = longType.getName();
            String propertyName = name + "s";
            Property property = routerConfig.findProperty(propertyName);

            if (property != null) {
                String values = property.getValue();

                if (values != null) {
                    List<String> valueStrs = Splitters.by(',').trim().split(values);
                    List<Integer> thresholds = new LinkedList<Integer>();

                    for (String valueStr : valueStrs) {
                        try {
                            thresholds.add(Integer.parseInt(valueStr));
                        } catch (Exception e) {
                            // ignore
                        }
                    }
                    if (!thresholds.isEmpty()) {
                        longConfigs.put(name, thresholds);
                    }
                }
            }
        }

        treeParser.refresh(startTypes, matchTypes);

        Property maxMetricProperty = routerConfig.findProperty("maxMetricTagValues");

        if (maxMetricProperty != null) {
            int maxMetricTagValues = Integer.parseInt(maxMetricProperty.getValue());

            if (maxMetricTagValues != MetricTagAggregator.MAX_KEY_SIZE) {
                MetricTagAggregator.MAX_KEY_SIZE = maxMetricTagValues;
            }
        }

        Property timeout = routerConfig.findProperty("clientConnectTimeout");

        if (timeout != null) {
            this.timeout = Integer.parseInt(timeout.getValue());
        }
    }

    private boolean refreshRouters(PropertyConfig routerConfig) {
        String newRouters = routerConfig.findProperty("routers").getValue();
        if ((routers == null) || (!routers.equals(newRouters))) {
            routers = newRouters;
            return true;
        }
        return false;
    }

    public void setSample(double sample) {
        samplingRate = sample;
    }

    private void storeServersByUrl(String url) {
        try {
            URL u = new URL(url);
            int httpPort = u.getPort();
            ApplicationEnvironment.storeServers(routers, httpPort);
        } catch (Exception e) {
            // ignore
        }
    }

}
