package cn.com.duiba.cat.model.configuration;

import cn.com.duiba.boot.netflix.eureka.EurekaHttpUtils;
import cn.com.duiba.cat.Cat;
import cn.com.duiba.cat.CatConstants;
import cn.com.duiba.cat.analyzer.MetricTagAggregator;
import cn.com.duiba.cat.message.spi.MessageTree;
import cn.com.duiba.cat.model.configuration.client.entity.Server;
import cn.com.duiba.cat.model.configuration.property.entity.Property;
import cn.com.duiba.cat.model.configuration.property.entity.PropertyConfig;
import cn.com.duiba.cat.util.Splitters;
import cn.com.duiba.cat.util.StringUtils;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Joiner;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.SetUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.*;

public class DefaultClientConfigService implements ClientConfigService {

    private static final Logger LOGGER = LoggerFactory.getLogger(DefaultClientConfigService.class);

    private static final DefaultClientConfigService instance     = new DefaultClientConfigService();
    private final        String                     domain;
    private              String                     routers;
    private              Set<Server>                servers;
    private volatile     double                     samplingRate = 1d;
    private volatile     boolean                    block        = false;
    private volatile     int                        timeout      = 1000;
    private final        MessageTreeTypeParser      treeParser   = new MessageTreeTypeParser();
    private final        Map<String, List<Integer>> longConfigs  = new LinkedHashMap<String, List<Integer>>();

    private DefaultClientConfigService() {
        this.domain = System.getProperty("cat.app.name");
    }

    public static DefaultClientConfigService getInstance() {
        return instance;
    }

    @Override
    public int getClientConnectTimeout() {
        return timeout;
    }

    @Override
    public String getDomain() {
        return this.domain;
    }

    @Override
    public int getLongConfigThreshold(String key) {
        List<Integer> values = longConfigs.get(key);
        int value;

        if (values != null && !values.isEmpty()) {
            value = values.get(0);
        } else {
            value = ProblemLongType.findByName(key).getThreshold();
        }

        return value;
    }

    @Override
    public int getLongThresholdByDuration(String key, int duration) {
        List<Integer> values = longConfigs.get(key);

        if (values != null) {
            for (int i = values.size() - 1; i >= 0; i--) {
                int userThreshold = values.get(i);

                if (duration >= userThreshold) {
                    return userThreshold;
                }
            }
        }

        return -1;
    }

    @Override
    public Set<Server> getServers() {
        if (servers == null) {
            refreshConfig();
        }
        return servers;
    }

    @Override
    public String getRouters() {
        if (routers == null) {
            refreshConfig();
        }
        return routers;
    }

    public double getSamplingRate() {
        return samplingRate;
    }

    public boolean isMessageBlock() {
        return block;
    }

    @Override
    public MessageType parseMessageType(MessageTree tree) {
        if (!tree.canDiscard()) {
            return MessageType.NORMAL_MESSAGE;
        } else {
            return treeParser.parseMessageType(tree);
        }
    }

    public void refreshConfig() {
        int retry = 0;
        int maxRetryCount = 3;
        while (retry < maxRetryCount) {
            try {
                innerRefreshConfig();
                break;
            } catch (Exception e) {
                retry++;
                LOGGER.error("error when refresh config", e);
            }
        }
    }

    private Map<String, JSONObject> discoverCatConsumerServers() {
        String eurekaServerUrl = System.getProperty("eureka.service-url");
        return EurekaHttpUtils.getUpIpInstanceMap(eurekaServerUrl, CatConstants.CAT_SERVER_CONSUMER);
    }

    private void innerRefreshConfig() {
        Map<String, JSONObject> catConsumerServers = this.discoverCatConsumerServers();

        Set<Server> newServers = new HashSet<>(catConsumerServers.size());
        List<String> newRouters = new ArrayList<>(catConsumerServers.size());

        for (Map.Entry<String, JSONObject> serverEntry : catConsumerServers.entrySet()) {
            String ip = serverEntry.getKey();
            JSONObject instance = serverEntry.getValue();
            int httpPort = instance.getJSONObject("port").getIntValue("$");
            int tcpPort = 2280;

            Server server = new Server(ip);
            newServers.add(server);

            server.setHttpPort(httpPort);
            server.setPort(tcpPort);
            server.setEnabled(true);

            newRouters.add(ip + ":" + tcpPort);
        }

        this.refreshServers(newServers);

        PropertyConfig routerConfig = new PropertyConfig();
        routerConfig.addProperty(new Property("startTransactionTypes").setValue(null));
        routerConfig.addProperty(new Property("multiInstances").setValue("false"));
        routerConfig.addProperty(new Property("matchTransactionTypes").setValue(null));
        routerConfig.addProperty(new Property("block").setValue("false"));
        routerConfig.addProperty(new Property("routers").setValue(Joiner.on(";").join(newRouters)));
        routerConfig.addProperty(new Property("sample").setValue("1.0"));

        if (this.refreshRouters(routerConfig)) {
            this.refreshInnerConfig(routerConfig);
        }
    }

    private void refreshInnerConfig(PropertyConfig routerConfig) {
        samplingRate = Double.parseDouble(routerConfig.findProperty("sample").getValue());

        if (samplingRate <= 0) {
            samplingRate = 0;
        }

        block = Boolean.parseBoolean(routerConfig.findProperty("block").getValue());

        if (block) {
            Cat.disable();
        } else {
            Cat.enable();
        }

        String multiInstancesConfig = routerConfig.findProperty("multiInstances").getValue();
        if (StringUtils.isNotEmpty(multiInstancesConfig)) {
            boolean multiInstances = Boolean.parseBoolean(multiInstancesConfig);

            if (multiInstances) {
                Cat.enableMultiInstances();
            } else {
                Cat.disableMultiInstances();
            }
        }

        String startTypes = routerConfig.findProperty("startTransactionTypes").getValue();
        String matchTypes = routerConfig.findProperty("matchTransactionTypes").getValue();

        for (ProblemLongType longType : ProblemLongType.values()) {
            final String name = longType.getName();
            String propertyName = name + "s";
            Property property = routerConfig.findProperty(propertyName);

            if (property != null) {
                String values = property.getValue();

                if (values != null) {
                    List<String> valueStrs = Splitters.by(',').trim().split(values);
                    List<Integer> thresholds = new LinkedList<Integer>();

                    for (String valueStr : valueStrs) {
                        try {
                            thresholds.add(Integer.parseInt(valueStr));
                        } catch (Exception e) {
                            // ignore
                        }
                    }
                    if (!thresholds.isEmpty()) {
                        longConfigs.put(name, thresholds);
                    }
                }
            }
        }

        treeParser.refresh(startTypes, matchTypes);

        Property maxMetricProperty = routerConfig.findProperty("maxMetricTagValues");

        if (maxMetricProperty != null) {
            int maxMetricTagValues = Integer.parseInt(maxMetricProperty.getValue());

            if (maxMetricTagValues != MetricTagAggregator.MAX_KEY_SIZE) {
                MetricTagAggregator.MAX_KEY_SIZE = maxMetricTagValues;
            }
        }

        Property timeout = routerConfig.findProperty("clientConnectTimeout");

        if (timeout != null) {
            this.timeout = Integer.parseInt(timeout.getValue());
        }
    }

    private boolean refreshRouters(PropertyConfig routerConfig) {
        String newRouters = routerConfig.findProperty("routers").getValue();
        if ((this.routers == null) || (!this.routers.equals(newRouters))) {
            this.routers = newRouters;
            return true;
        }
        return false;
    }

    private void refreshServers(Set<Server> newServers) {
        if (this.servers != null && newServers != null) {
            Server s1 = new ArrayList<>(this.servers).get(0);
            Server s2 = new ArrayList<>(newServers).get(0);
            System.out.println(s1.equals(s2));
            System.out.println(this.servers.contains(s1));
            System.out.println(this.servers.contains(s2));
            System.out.println(newServers.contains(s2));
            System.out.println(newServers.contains(s2));
            System.out.println(this.servers.equals(newServers));
            System.out.println(this.servers.containsAll(newServers));
        }

        if (this.servers == null || !SetUtils.isEqualSet(this.servers, newServers)) {
            this.servers = newServers;
        }
    }

    public void setSample(double sample) {
        samplingRate = sample;
    }

}
