package cn.com.duiba.galaxy.sdk.component;

import cn.com.duiba.wolf.utils.DateUtils;

import java.time.DayOfWeek;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.ZoneId;
import java.time.temporal.TemporalAdjusters;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;

/**
 * @program: projectx
 * @description: 组件限制 - 时间粒度枚举类
 * @author: Simba
 * @create: 2021-02-24 13:50
 **/
public enum TimeUnitEnum {
    NO_LIMIT(0, "不限制"),

    DAY(1, "日"),

    WEEK(2, "周"),

    MONTH(3, "月"),

    FOREVER(4, "永久"),

    CUSTOM(5, "自定义"),

    ;

    private int type;

    private String desc;

    TimeUnitEnum(int type, String desc) {
        this.type = type;
        this.desc = desc;
    }

    public static TimeUnitEnum toEnum(int type) {
        for (TimeUnitEnum p : values()) {
            if (p.type == type) {
                return p;
            }
        }
        return null;
    }

    public int getType() {
        return type;
    }

    public String getDesc() {
        return desc;
    }


    /**
     * 获取限制的到期时长
     *
     * ! 不支持自定义周期的计算
     *
     * @return
     */
    public Long getNowToEndTime() {
        Date now = new Date();
        if (this == DAY) {
            return DateUtils.getDayEndTime(now).getTime()-now.getTime();
        } else if (this == WEEK) {
            return getWeekLastDay(now).getTime()-now.getTime();
        } else if (this == MONTH) {
            return getMonthLastDay(now).getTime()-now.getTime();
        } else if (this == FOREVER) {
            return DateUtils.daysAddOrSub(now,365*6).getTime()-now.getTime();
        } else {
            return null;
        }
    }


    /**
     * 周期开始时间
     * 永久是""
     * 自定义和不限制是null
     * @return
     */
    public String geStartTime() {

        LocalDate now = LocalDate.now();
        if (this == DAY) {
            return now.toString();
        } else if (this == WEEK) {
            return now.minusDays(now.getDayOfWeek().getValue()-1).toString();
        } else if (this == MONTH) {
            return now.withDayOfMonth(1).toString();
        } else if (this == FOREVER) {
            return "";
        } else {
            return null;
        }
    }


    /**
     * 获得除自身之外的其他限制类型  仅限 日周月永久
     * @return
     */
    public List<TimeUnitEnum> geOtherTimeUnit() {
        List<TimeUnitEnum> effectiveTimeUnit= new ArrayList<>(Arrays.asList(TimeUnitEnum.DAY,TimeUnitEnum.WEEK,TimeUnitEnum.MONTH,TimeUnitEnum.FOREVER));
        effectiveTimeUnit.remove(this);
        return effectiveTimeUnit;
    }


    /**
     * 获得除自身之外的其他限制类型  仅限 日周月永久
     * @return
     */
    public List<TimeUnitEnum> geOtherAwardTimeUnit() {
        List<TimeUnitEnum> effectiveTimeUnit= new ArrayList<>(Arrays.asList(TimeUnitEnum.DAY,TimeUnitEnum.WEEK,TimeUnitEnum.MONTH));
        effectiveTimeUnit.remove(this);
        return effectiveTimeUnit;
    }


    private static Date getWeekLastDay(Date date) {
        LocalDate localDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        LocalDateTime localDateTime = LocalDateTime.of(localDate.with(DayOfWeek.SUNDAY), LocalTime.MAX);
        return Date.from(localDateTime.atZone(ZoneId.systemDefault()).toInstant());
    }

    private static Date getMonthLastDay(Date date) {
        LocalDate localDate = date.toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
        LocalDateTime lastDateTime = LocalDateTime.of(localDate.with(TemporalAdjusters.lastDayOfMonth()), LocalTime.MAX);
        return Date.from(lastDateTime.atZone(ZoneId.systemDefault()).toInstant());
    }


}
