package cn.com.duiba.galaxy.sdk.api.prize;

import cn.com.duiba.galaxy.common.utils.PageList;
import cn.com.duiba.galaxy.sdk.api.base.AutowiredApi;
import cn.com.duiba.galaxy.sdk.api.prize.inner.Option;
import cn.com.duiba.galaxy.sdk.api.prize.inner.StrategyResult;
import cn.com.duiba.galaxy.sdk.api.prize.inner.StrategyTypeEnum;
import cn.com.duiba.galaxy.sdk.api.prize.inner.UserSpRecordNew;

import java.time.LocalDateTime;
import java.util.List;
import java.util.Map;

/**
 * @author jiaxz
 * @since 2022-12-30
 */
public interface PrizeApi extends AutowiredApi {

    /**
     * 可以指定一个用户进行策略发奖
     * 注意：如果你得活动是模板活动，那么只能指定发奖规则(StrategyTypeEnum)
     * 如何获取StrategyTypeEnum？？getEnableStrategyType
     *
     * @param userId
     * @param prizeStrategyId 发奖策略ID
     * @return 奖项
     */
    StrategyResult sendPrizeWithStrategy(String userId, String prizeStrategyId);

    /**
     * 发放道具奖品  不会在发奖记录中记录  只会在道具流水中记录
     *
     * @param userId
     * @param spId
     * @param spNum
     * @return
     */
    StrategyResult sendSpPrizeNoLog(Long userId, String spId, Long spNum);


    /**
     * 指定奖品发奖
     * 注意：本方法是忽略发奖规则的概率策略的  指定中奖奖品
     *
     * @param userId
     * @param ruleId
     * @param optionId
     * @return
     */
    StrategyResult sendPrizeWithOptionId(Long userId, String ruleId, Long optionId);

    /**
     * 查询我的某个道具的数量
     *
     * @param spId
     * @return
     */
    Long getStageProperty(Long userId, String spId);

    /**
     * 查询我所有的道具的数量信息
     *
     * @return key:道具id，value:拥有的道具数
     */
    Map<String, Long> getAllStageProperty(Long userId);


    /**
     * 注意：慎用，quantity传long,避免sp_record sum报错
     * <p>
     * 给用户发放道具
     *
     * @param userId   userId
     * @param spId     道具ID
     * @param quantity 道具个数
     * @return 记录ID
     */
    Long giveStageProperty(Long userId, String spId, long quantity);


    /**
     * 消耗某个用户的道具
     *
     * @param userId   userId
     * @param spId     spId
     * @param quantity 数量
     * @return 消耗是否成功
     */
    Boolean consumeStageProperty(Long userId, String spId, long quantity);

    /**
     * 查询某人所有的道具的数量信息
     *
     * @param userId 需要查询的用户
     * @return key:道具id，value:拥有的道具数
     */
    Map<String, Long> getSomebodyAllStageProperty(Long userId);

    /**
     * 查询某些用户的某个道具数
     *
     * @param userIds 用户id最多支持1000人，超过抛异常
     * @param spId    sp id
     * @return map user id -> sp count
     */
    Map<Long, Long> getSpByUserIds(List<String> userIds, String spId);

    /**
     * 查询某人的某个道具的数量
     *
     * @param userId 需要查询的用户
     * @param spId   道具id
     * @return
     */
    Long getSomebodyStageProperty(Long userId, String spId);


    /**
     * 分页页查询用户道具记录
     * ⚠️ PageSize 不能大于200
     *
     * @param userId
     * @param spId
     * @param pageNum
     * @param pageSize
     * @return
     */
    PageList<UserSpRecordNew> pageQuerySpRecord(Long userId, String spId, long pageNum, long pageSize);

    /**
     * 根据时间范围分页页查询用户道具记录
     * ⚠️ PageSize 不能大于200
     *
     * @param userId
     * @param spId
     * @param pageNum
     * @param pageSize
     * @return
     */
    PageList<UserSpRecordNew> pageQuerySpRecord(Long userId, String spId, long pageNum, long pageSize, LocalDateTime startTime, LocalDateTime endTime);


    /**
     * 根据场景id查询对应的发奖规则
     *
     * @param sceneId
     * @return
     */
    List<String> queryRuIdsBySceneId(String sceneId);

    /**
     * 获取某个发奖策略下面的奖项列表
     *
     * @param strategyId 发奖策略id
     * @param forInner   (true 内部使用，会暴露中奖概率和库存字段)
     * @return list
     */
    List<Option> queryOptions(String strategyId, boolean forInner);


    /**
     * 根据场景id发奖
     *
     * @param sceneId
     * @return
     */
    StrategyResult sendPrizeBySceneId(Long userId, String sceneId);

    /**
     * 根据场景id发奖
     *
     * @param sceneId
     * @return
     */
    StrategyResult sendMyPrizeBySceneId(String sceneId);


    /**
     * 获取启用策略类型
     * 注意:目前只针对模板活动查询已启用的发奖规则
     *
     * @return {@link List}<{@link String}>
     */
    StrategyTypeEnum getEnableStrategyType();


    String getStrategyName(Long projectId, String ruleId);


    Option getStrategyFirstOption(Long projectId, String ruleId);


    /**
     * 获得用户奖品记录
     *
     * @param ignoreAddCredits 是否忽略增加积分的奖品
     * @return
     */
    List<UserPrizeRecordVO> getUserPrizeRecord(boolean ignoreAddCredits);


    /**
     * 谢谢参与奖品
     *
     * @return
     */
    StrategyResult getThanksPrize();
}
