package cn.com.duiba.live.activity.center.api.util.citic.mgm;

import cn.com.duiba.live.activity.center.api.bean.citic.mgm.MgmM2InfoBean;
import cn.com.duiba.live.activity.center.api.enums.citic.mgm.MgmM2StatusEnum;
import cn.hutool.core.date.DateTime;
import cn.hutool.core.date.DateUnit;
import cn.hutool.core.date.DateUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.bouncycastle.crypto.digests.SM3Digest;
import org.bouncycastle.util.encoders.Hex;

import java.nio.charset.StandardCharsets;
import java.util.Date;

/**
 * @author zhuenhui
 * @date 2022/05/24 17:31
 * @description 中信MGM - 工具类
 */
@Slf4j
public class MgmUtils {

    /**
     * 新户判断字段
     */
    private static final String NEW_CUST_FLAG = "是";

    private MgmUtils() {
    }

    /**
     * 计算过期时间
     *
     * @param date     时间
     * @param interval 过期时间间隔 单位：天
     * @return 选择时间
     */
    public static Date calExpiredTime(Date date, int interval) {
        if (date == null) {
            return null;
        }
        // 时间向后增加指定天数
        DateTime dateTime = DateUtil.offsetDay(date, interval);
        // 获取当天的最后一秒
        DateTime endOfDay = DateUtil.endOfDay(dateTime);
        // 兼容基础Date
        return endOfDay.toJdkDate();
    }

    /**
     * 验证是否过期
     *
     * @param date     需要验证的日期
     * @param interval 时间间隔 单位：天
     * @return true - 过期, false - 没过期
     */
    public static boolean verifyExpired(Date date, int interval) {
        if (date == null) {
            return true;
        }
        // 计算到期时间的最后一秒
        Date expiredDate = calExpiredTime(date, interval);
        // 判断是否过期
        return new Date().after(expiredDate);
    }

    /**
     * 计算m2状态
     *
     * @param bean              m2信息
     * @param m2InvalidInterval m2失效间隔
     * @return 类型
     * {@link MgmM2StatusEnum}
     */
    public static Integer calM2Status(MgmM2InfoBean bean, Integer m2InvalidInterval) {
        if (bean == null) {
            return null;
        }
        // 获取用户状态
        MgmM2StatusEnum m2Status = MgmM2StatusEnum.getByCardDesc(bean.getCardStatusDesc());
        if (m2Status == null) {
            log.warn("citic mgm, not match m2 status, bean:{}", bean);
            return MgmM2StatusEnum.UNKNOWN.getStatus();
        }
        // 审核中、待面签、已拒绝（这三种状态没有新户标识）
        if (MgmM2StatusEnum.NO_NEW_CUST_FLAG_LIST.contains(m2Status.getStatus())) {
            return m2Status.getStatus();
        }
        // 不是新户
        if (!NEW_CUST_FLAG.equals(bean.getNewCustFlag())) {
            return MgmM2StatusEnum.NOT_MATCH.getStatus();
        }
        // 如果完成首刷，则检测是否失效
        if (MgmM2StatusEnum.CONSUME.equals(m2Status) && isConsumeInvalid(bean.getCreateDate(), bean.getCardFirstPurchaseDte(), m2InvalidInterval)) {
            return MgmM2StatusEnum.INVALID.getStatus();
        }
        return m2Status.getStatus();
    }

    private static boolean isConsumeInvalid(Date createTime, Date consumeTime, Integer m2InvalidInterval) {
        if (createTime == null || consumeTime == null || m2InvalidInterval == null) {
            return true;
        }
        // 计算相隔天数
        long betweenDays = DateUtil.between(createTime, consumeTime, DateUnit.DAY, true);
        // 判断是否失效
        return betweenDays > m2InvalidInterval;
    }

    /**
     * 加密
     *
     * @param data   原始数据
     * @param pubKey MGM公钥
     * @return 加密数据
     */
    public static String encode(String data, String pubKey) {
        if (StringUtils.isBlank(data)) {
            return "";
        }
        try {
            // 中信工具类加密数据
            byte[] decMsg = KzxSM2Utils.encrypt(pubKey, data.getBytes(StandardCharsets.UTF_8));
            // 返回加密后的数据
            return bytes2hex02(decMsg);
        } catch (Exception e) {
            log.warn("citic mgm, encode data fail, data:{}", data, e);
            return "";
        }
    }

    public static String bytes2hex02(byte[] bytes) {
        StringBuilder sb = new StringBuilder();
        String tmp = null;
        for (byte b : bytes) {
            // 将每个字节与0xFF进行与运算，然后转化为10进制，然后借助于Integer再转化为16进制
            tmp = Integer.toHexString(0xFF & b);
            // 每个字节8为，转为16进制标志，2个16进制位
            if (tmp.length() == 1) {
                tmp = "0" + tmp;
            }
            sb.append(tmp);
        }
        return sb.toString();
    }

    /**
     * 解密
     *
     * @param encodeData 加密数据
     * @param priKey     兑吧私钥
     * @return 解密数据
     */
    public static String decode(String encodeData, String priKey) {
        if (StringUtils.isBlank(encodeData)) {
            return "";
        }
        try {
            // 中信工具类解密数据
            byte[] decMsg = KzxSM2Utils.decrypt(priKey, Hex.decode(encodeData));
            // 返回解密后的数据
            return new String(decMsg, StandardCharsets.UTF_8);
        } catch (Exception e) {
            log.warn("citic mgm, decode data fail, encodeData:{}", encodeData, e);
            return "";
        }
    }

    /**
     * 获取签名
     *
     * @param encodeData 加密数据
     * @param priKey     兑吧私钥
     * @return 签名
     */
    public static String sign(String encodeData, String priKey) {
        if (StringUtils.isBlank(encodeData)) {
            return "";
        }
        try {
            // 加密数据摘要
            String digest = sm3(encodeData);
            // 签名
            return KzxSM2Utils.sign(priKey, digest.getBytes(StandardCharsets.UTF_8));
        } catch (Exception e) {
            log.warn("citic mgm, sign encodeData fail, encodeData:{}", encodeData, e);
            return "";
        }
    }

    /**
     * 验签
     *
     * @param encodeData 原始数据
     * @param sign       签名
     * @param pubKey     MGM公钥
     * @return 加密数据
     */
    public static boolean verify(String encodeData, String sign, String pubKey) {
        if (StringUtils.isBlank(encodeData)) {
            return false;
        }
        try {
            // 加密数据摘要
            String digest = sm3(encodeData);
            // 使用中信工具类验证
            return KzxSM2Utils.verify(pubKey, digest.getBytes(StandardCharsets.UTF_8), Hex.decode(sign));
        } catch (Exception e) {
            log.warn("citic mgm, verify data fail, encodeData:{}, sign:{}", encodeData, sign, e);
            return false;
        }
    }

    /**
     * sm3摘要
     *
     * @param encodeData 加密数据
     * @return 摘要信息
     */
    private static String sm3(String encodeData) {
        byte[] md = new byte[32];
        byte[] msg1 = encodeData.getBytes(StandardCharsets.UTF_8);
        SM3Digest sm3 = new SM3Digest();
        sm3.update(msg1, 0, msg1.length);
        sm3.doFinal(md, 0);
        return new String(Hex.encode(md));
    }

}
