package cn.com.duiba.live.tool.util;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.module.SimpleModule;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.math.BigDecimal;

/**
 * Created by hong
 * time 2021/9/23.
 * @description: Json序列化工具
 */
public class JsonUtils {

    private static final Logger log = LoggerFactory.getLogger(JsonUtils.class);

    public static ObjectMapper objectMapper = new ObjectMapper();

    static {
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        objectMapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);

        // 序列化Long/long -> String，避免前端异常
        SimpleModule simpleModule = new SimpleModule();
        simpleModule.addSerializer(new StringSerializer<>(Long.TYPE));
        simpleModule.addSerializer(new StringSerializer<>(Long.class));
        simpleModule.addDeserializer(BigDecimal.class, new BigDecimalDeserializer());

        objectMapper.registerModule(simpleModule);
    }

    public static String toJsonString(Object object) {
        try {
            return objectMapper.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            log.error("JSON序列化失败, objectClass:{}", object.getClass().getName(), e);
            throw new RuntimeException(e);
        }
    }

    public static <T> T parseObject(String string, Class<T> tClass) {
        if (string == null || tClass == null) {
            log.warn("JSON反序列化参数为空, string:{}, tClass:{}", string, tClass);
            throw new RuntimeException("JSON反序列化参数为空");
        }

        try {
            return objectMapper.readValue(string, tClass);
        } catch (IOException e) {
            log.error("JSON反序列化失败, string:{}, tClass:{}", string, tClass.getName(), e);
            throw new RuntimeException(e);
        }
    }

    public static <T> T parseObject(String string, TypeReference<T> valueTypeRef) {
        if (string == null || valueTypeRef == null) {
            log.warn("JSON反序列化参数为空, string:{}, valueTypeRef:{}", string, valueTypeRef);
            throw new RuntimeException("JSON反序列化参数为空");
        }

        try {
            return objectMapper.readValue(string, valueTypeRef);
        } catch (IOException e) {
            log.error("JSON反序列化失败, string:{}, valueTypeRef:{}", string, valueTypeRef, e);
            throw new RuntimeException(e);
        }
    }

}
