package cn.com.duiba.live.tool.exception;

import cn.com.duiba.boot.exception.BizException;
import com.alibaba.fastjson.JSON;
import org.apache.commons.io.IOUtils;
import org.springframework.boot.web.servlet.error.ErrorAttributes;
import org.springframework.core.Ordered;
import org.springframework.http.HttpStatus;
import org.springframework.web.context.request.ServletWebRequest;
import org.springframework.web.context.request.WebRequest;
import org.springframework.web.servlet.HandlerExceptionResolver;
import org.springframework.web.servlet.ModelAndView;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.lang.reflect.UndeclaredThrowableException;
import java.nio.charset.StandardCharsets;
import java.util.Map;

/**
 * 这个HandlerExceptionResolver(全局mvc异常处理)用于对BizException类进行特殊处理，返回的错误json字符串中多加入一个code字段
 * <br/>
 * 并且这个异常不会打印到日志中。
 * Created by hong
 * time 2021/3/5.
 */
public class BizExceptionResolverCustom implements HandlerExceptionResolver, Ordered {

    private ErrorAttributes errorAttributes;

    public BizExceptionResolverCustom() {
    }

    public BizExceptionResolverCustom(ErrorAttributes errorAttributes) {
        this.errorAttributes = errorAttributes;
    }

    @Override
    public int getOrder() {
        return Ordered.LOWEST_PRECEDENCE - 10;
    }

    @Override
    public ModelAndView resolveException(HttpServletRequest request, HttpServletResponse response, Object handler, Exception ex) {
        //只对BizException进行特殊处理, 比BasicErrorController的处理多返回一个code参数

        Throwable t = ex;
        if(t instanceof UndeclaredThrowableException){
            t = ((UndeclaredThrowableException) t).getUndeclaredThrowable();
        }

        if(t instanceof BizException){
            Map<String, Object> body = getErrorAttributes(request, false);
            body.put("status", HttpStatus.INTERNAL_SERVER_ERROR.value());
            body.put("error", HttpStatus.INTERNAL_SERVER_ERROR.getReasonPhrase());
            if(((BizException) t).getCode() != null) {
                body.put("code", ((BizException) t).getCode());
            }
            body.put("exception", t.getClass().getName());
            body.put("message", t.getMessage());
            String jsonStr = JSON.toJSONString(body);
            try {
                response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
                response.setContentType("application/json;charset=UTF-8");
                IOUtils.write(jsonStr, response.getOutputStream(), StandardCharsets.UTF_8);
                IOUtils.closeQuietly(response.getOutputStream());
            } catch (IOException e) {
                throw new RuntimeException(e.getMessage(), e);
            }
            return new ModelAndView();
        }

        if(t instanceof DuibaLiveBizException){
            Map<String, Object> body = getErrorAttributes(request, false);
            body.put("status", HttpStatus.INTERNAL_SERVER_ERROR.value());
            body.put("error", HttpStatus.INTERNAL_SERVER_ERROR.getReasonPhrase());
            if(((DuibaLiveBizException) t).getCode() != null) {
                body.put("code", ((DuibaLiveBizException) t).getCode());
            }
            body.put("exception", t.getClass().getName());
            body.put("message", t.getMessage());
            String jsonStr = JSON.toJSONString(body);
            try {
                response.setStatus(HttpStatus.INTERNAL_SERVER_ERROR.value());
                IOUtils.write(jsonStr, response.getOutputStream(), StandardCharsets.UTF_8);
                IOUtils.closeQuietly(response.getOutputStream());
            } catch (IOException e) {
                throw new RuntimeException(e.getMessage(), e);
            }
            return new ModelAndView();
        }

        //返回null表示不处理.
        return null;
    }

    private Map<String, Object> getErrorAttributes(HttpServletRequest request,
                                                   boolean includeStackTrace) {
        WebRequest webRequest = new ServletWebRequest(request);
        return this.errorAttributes.getErrorAttributes(webRequest,
                                                       includeStackTrace);
    }
}
