package cn.com.duiba.live.tool.util;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.StdScalarSerializer;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;

/**
 * Created by hong
 * time 2021/9/23.
 * @description 日志JSON序列化工具
 * @warning 这个类只应该在记录日志的时候使用
 * @see JsonUtils 与JsonUtils的区别在于，LogJsonUtils在遇到长字符串的时候，会将其进行裁剪。
 * 解决了在记录日志的时候，由于对象中有Base64/byte[]等长字符串的情况下日志特别大的问题。
 */
public class LogJsonUtils {

        private static final Logger log = LoggerFactory.getLogger(LogJsonUtils.class);

    public static String toJsonString(Object object) {
        try {
            return objectMapper.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            log.error("JSON序列化失败, objectClass:{}", object.getClass().getName(), e);
            return StringUtils.EMPTY;
        }
    }

    public static ObjectMapper objectMapper = new ObjectMapper();

    static {
        objectMapper.setSerializationInclusion(JsonInclude.Include.NON_NULL);
        objectMapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);

        SimpleModule simpleModule = new SimpleModule();
        // 避免出现长String如Base64的情况下让日志很大
        simpleModule.addSerializer(new StringSerializer());
        // 避免出现byte[]是序列化导致日志很大
        simpleModule.addSerializer(new ByteSerializer());
        objectMapper.registerModule(simpleModule);
    }

    private static class StringSerializer extends StdScalarSerializer<String> {

        StringSerializer() {
            super(String.class);
        }

        @Override
        public void serialize(String value, JsonGenerator gen, SerializerProvider provider) throws IOException {
            if (value == null) {
                gen.writeString((String) null);
                return;
            }
            if (value.length() <= 1024) {
                gen.writeString(value);
                return;
            }

            gen.writeString(value.substring(0, 10) + "..." + value.substring(value.length() - 10));
        }
    }

    private static class ByteSerializer extends StdScalarSerializer<byte[]> {

        ByteSerializer() {
            super(byte[].class);
        }

        @Override
        public void serialize(byte[] value, JsonGenerator gen, SerializerProvider provider) throws IOException {
            if (value == null) {
                gen.writeString((String) null);
                return;
            }
            gen.writeString("byte[]:" + value.length);
        }
    }

}
